<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Stadium;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Yajra\DataTables\Facades\DataTables;

class StadiumController extends Controller
{
     public function index()
    {
        return view('admin.stadiums');
    }
    // List (DataTable)
    public function getStadiums(Request $request)
    {
        if ($request->ajax()) {
            $data = Stadium::select('id', 'name', 'city', 'country', 'layout_image');
            return DataTables::of($data)
                ->addIndexColumn()
                ->addColumn('layout_image', function ($row) {
                    return $row->layout_image ? '<img src="' . asset($row->layout_image) . '" width="60">' : '';
                })
                ->addColumn('action', function ($row) {
                    return '
                        <button class="btn btn-sm btn-primary editStadium" data-id="' . $row->id . '">Edit</button>
                        <button class="btn btn-sm btn-danger deleteStadium" data-id="' . $row->id . '">Delete</button>
                    ';
                })
                ->rawColumns(['layout_image', 'action'])
                ->make(true);
        }
    }

    // Store
    public function store(Request $request)
    {
        $request->validate([
            'name'    => 'required|string|max:255',
            'city'    => 'nullable|string|max:255',
            'country' => 'nullable|string|max:255',
            'photo'   => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $imagePath = null;

        if ($request->hasFile('photo')) {
            $image = $request->photo;
            $imageName = time() . '-' . rand(100, 999) . '.' . $image->extension();
            $uploadPath = public_path('assets/upload/stadiums/');

            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0755, true);
            }

            $image->move($uploadPath, $imageName);
            $imagePath = 'assets/upload/stadiums/' . $imageName;
        }

        Stadium::create([
            'name'         => ucfirst($request->name),
            'city'         => $request->city,
            'country'      => $request->country,
            'layout_image' => $imagePath,
        ]);

        return response()->json(['success' => 'Stadium added successfully!']);
    }

    // Update
    public function update(Request $request, $id)
    {
        $stadium = Stadium::findOrFail($id);

        $request->validate([
            'name'    => 'required|string|max:255',
            'city'    => 'nullable|string|max:255',
            'country' => 'nullable|string|max:255',
            'photo'   => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $imagePath = $stadium->layout_image;

        if ($request->hasFile('photo')) {
            $image = $request->photo;
            $imageName = time() . '-' . rand(100, 999) . '.' . $image->extension();
            $uploadPath = public_path('assets/upload/stadiums/');

            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0755, true);
            }

            if ($stadium->layout_image && File::exists(public_path($stadium->layout_image))) {
                File::delete(public_path($stadium->layout_image));
            }

            $image->move($uploadPath, $imageName);
            $imagePath = 'assets/upload/stadiums/' . $imageName;
        }

        $stadium->update([
            'name'         => ucfirst($request->name),
            'city'         => $request->city,
            'country'      => $request->country,
            'layout_image' => $imagePath,
        ]);

        return response()->json(['success' => 'Stadium updated successfully!']);
    }

    // Delete
    public function destroy($id)
    {
        $stadium = Stadium::findOrFail($id);

        if ($stadium->layout_image && File::exists(public_path($stadium->layout_image))) {
            File::delete(public_path($stadium->layout_image));
        }

        $stadium->delete();

        return response()->json(['success' => 'Stadium deleted successfully!']);
    }
}
