<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\HomeSection;
use App\Models\Category;
use App\Models\SubCategory;
use Yajra\DataTables\Facades\DataTables;

class HomeSectionController extends Controller
{
    public function index()
    {
        $categories = Category::get();
        $subcategories = SubCategory::where('is_active', 1)->get();

        return view('admin.home_section', compact('categories', 'subcategories'));
    }

    public function list(Request $request)
    {
        $data = HomeSection::latest()->get();

        return Datatables::of($data)
            ->addIndexColumn()

            ->addColumn('source', function ($row) {
                if ($row->source_type === 'category') {
                    return Category::find($row->source_id)->name ?? '-';
                }
                return SubCategory::find($row->source_id)->name ?? '-';
            })

            ->addColumn('status', function ($row) {
                $checked = $row->status ? 'checked' : '';
                return '<input type="checkbox" class="toggleStatus" data-id="' . $row->id . '" ' . $checked . '>';
            })
            ->addColumn('position', function ($row) {
                return $row->position;
            })

            ->addColumn('action', function ($row) {
                return '
                <button class="btn btn-sm btn-primary editHomeSection"
                    data-id="' . $row->id . '"
                    data-title="' . $row->title . '"
                    data-position="' . $row->position . '"
                    data-source_type="' . $row->source_type . '"
                    data-source_id="' . $row->source_id . '"
                    data-ui_type="' . $row->ui_type . '"
                >Edit</button>

                <button class="btn btn-sm btn-danger deleteHomeSection"
                    data-id="' . $row->id . '"
                >Delete</button>';
            })

            ->rawColumns(['status', 'action'])
            ->make(true);
    }

    public function store(Request $request)
    {
        $rules = [
            'title' => 'required',
            'source_type' => 'required|in:category,subcategory,all_category,banner,seasonal_banner',
            'ui_type' => 'required'
        ];

        if ($request->source_type !== 'all_category' && $request->source_type !== 'banner'&& $request->source_type !== 'seasonal_banner') {
            $rules['source_id'] = 'required';
        }

        $request->validate($rules);

        HomeSection::create([
            'title'       => $request->title,
            'source_type' => $request->source_type,
            'source_id'   => $request->source_type === 'all_category' ? null : $request->source_id,
            'ui_type'     => $request->ui_type,
            'position' =>    $request->position
        ]);

        return response()->json(['success' => 'Home section added successfully']);
    }


    public function update(Request $request, $id)
    {
        $section = HomeSection::findOrFail($id);

        $section->update($request->all());

        return response()->json(['success' => 'Home section updated successfully']);
    }

    public function destroy($id)
    {
        HomeSection::findOrFail($id)->delete();
        return response()->json(['success' => 'Home section deleted']);
    }

    public function toggleStatus($id)
    {
        $section = HomeSection::findOrFail($id);
        $section->status = !$section->status;
        $section->save();

        return response()->json(['success' => 'Status updated']);
    }
}
