<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\Products;
use App\Models\Banner;
use App\Models\SubCategory;
use App\Models\Address;
use App\Models\Order;
use App\Models\OrderItems;
use App\Models\OrderDiscount;
use Illuminate\Support\Facades\DB;
use App\Models\Coupon;
use Carbon\Carbon;
use App\Models\Review;
use Barryvdh\DomPDF\Facade\Pdf;

class OrderController extends Controller
{
    // public function placeOrder(Request $request)
    // {
    //     try {
    //         // 🔍 Validate input
    //         $request->validate([
    //             'address_id'       => 'required|exists:addresses,id',
    //             'delivery_details' => 'nullable|array',
    //             'discount_amount'  => 'nullable|numeric|min:0',
    //             'delivery_charge'  => 'nullable|numeric|min:0',
    //             'items'            => 'required|array|min:1',
    //             'items.*.product_id' => 'required|exists:products,id',
    //             'items.*.quantity'   => 'required|integer|min:1',
    //         ]);


    //         $order = DB::transaction(function () use ($request) {

    //             // Generate unique order number
    //             $orderNumber = 'ORD-' . strtoupper(uniqid());

    //             // Create order with default values
    //             $order = Order::create([
    //                 'order_number'    => $orderNumber,
    //                 'user_id'         => auth()->user()->id,
    //                 'address_id'      => $request->address_id,
    //                 'subtotal'        => 0,
    //                 'discount_amount' => $request->discount_amount ?? 0,
    //                 'delivery_charge' => $request->delivery_charge ?? 0,
    //                 'total_amount'    => 0,
    //                 'delivery_details' => $request->delivery_details ? json_encode($request->delivery_details) : null,
    //                 'payment_status'   => 'pending',   // enum field
    //                 'order_status'     => 'pending',   // enum field
    //             ]);




    //             $subtotal = 0;

    //             // Insert order items
    //             foreach ($request->items as $item) {
    //                 $product = Products::find($item['product_id']);

    //                 if (!$product) {
    //                     throw new \Exception("Product ID {$item['product_id']} not found!");
    //                 }

    //                 // $price = $product->discount_price ?? $product->price;
    //                 $price = $item['price'];
    //                 OrderItems::create([
    //                     'order_id'   => $order->id,
    //                     'product_id' => $product->id,
    //                     'product_type' => $item['product_type'],
    //                     'quantity'   => $item['quantity'],
    //                     'price'      => $price,
    //                     'total_price' => $price * $item['quantity'],
    //                     'custom_fields' => $item['custom_fields'] ? json_encode($item['custom_fields']) : null,
    //                 ]);

    //                 $subtotal += $price * $item['quantity'];
    //             }


    //             // Final amount
    //             $totalAmount = $subtotal - $order->discount_amount + $order->delivery_charge;

    //             // Update order final values
    //             $order->update([
    //                 'subtotal'     => $subtotal,
    //                 'total_amount' => $totalAmount,
    //             ]);

    //             return $order;
    //         });

    //         return response()->json([
    //             'status'  => true,
    //             'message' => 'Order placed successfully!',
    //             'order'   => $order,
    //         ], 201);
    //     } catch (\Exception $e) {

    //         return response()->json([
    //             'status'  => false,
    //             'message' => $e->getMessage(),
    //         ], 500);
    //     }
    // }



public function placeOrder(Request $request)
{
    try {

        $request->validate([
            'address_id' => 'required|exists:addresses,id',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
        ]);

        $order = DB::transaction(function () use ($request) {

            $orderNumber = 'ORD-' . strtoupper(uniqid());

            $order = Order::create([
                'order_number' => $orderNumber,
                'user_id' => auth()->id(),
                'address_id' => $request->address_id,
                'subtotal' => 0,
                'discount_amount' => $request->discount_amount ?? 0,
                'delivery_charge' => $request->delivery_charge ?? 0,
                'total_amount' => 0,
                'delivery_details' => $request->delivery_details
                    ? json_encode($request->delivery_details)
                    : null,
                'payment_status' => 'pending',
                'order_status' => 'pending',
            ]);

            $subtotal = 0;

            foreach ($request->items as $item) {
                $product = Products::findOrFail($item['product_id']);
                $price = $item['price'];

                OrderItems::create([
                    'order_id' => $order->id,
                    'product_id' => $product->id,
                    'product_type' => $item['product_type'] ?? null,
                    'quantity' => $item['quantity'],
                    'price' => $price,
                    'total_price' => $price * $item['quantity'],
                    'custom_fields' => isset($item['custom_fields'])
                        ? json_encode($item['custom_fields'])
                        : null,
                ]);

                $subtotal += $price * $item['quantity'];
            }

            $totalAmount = $subtotal - $order->discount_amount + $order->delivery_charge;

            $order->update([
                'subtotal' => $subtotal,
                'total_amount' => $totalAmount,
            ]);

            /* ============================
               GENERATE & SAVE INVOICE PDF
               ============================ */

            $invoiceDir = public_path('assets/upload/invoice/');
            if (!file_exists($invoiceDir)) {
                mkdir($invoiceDir, 0755, true);
            }

            $pdf = Pdf::loadView('admin.invoice', [
                'order' => $order->load('items.product','user')
            ]);

            $fileName = 'invoice_'.$order->order_number.'.pdf';
            $pdf->save($invoiceDir . $fileName);

            $order->update([
                'invoice_path' => 'assets/upload/invoice/' . $fileName
            ]);

            return $order;
        });

        return response()->json([
            'status' => true,
            'message' => 'Order placed & invoice generated',
            'order_id' => $order->id,
       //     'invoice_url' => asset($order->invoice_path)
        ], 201);

    } catch (\Exception $e) {
        return response()->json([
            'status' => false,
            'message' => $e->getMessage()
        ], 500);
    }
}

    public function addDeliveryAddress(Request $request)
    {
        try {
            // 🧾 Validation
            $request->validate([
                'recipient_name' => 'required|string|max:100',
                'recipient_phone' => 'required|string|max:15',
                'alternate_phone' => 'nullable|string|max:15',
                'address' => 'required|string',
                'landmark' => 'nullable|string',
                'pincode' => 'required|string|max:10',
                'country' => 'required|string|max:100',
            ]);

            // 💾 Save
            $address = Address::create([
                'user_id' => auth()->user()->id,
                'recipient_name' => $request->recipient_name,
                'recipient_phone' => $request->recipient_phone,
                'alternate_phone' => $request->alternate_phone,
                'address' => $request->address,
                'landmark' => $request->landmark,
                'pincode' => $request->pincode,
                'country' => $request->country,
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Delivery address added successfully',
                'data' => $address,
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage(),
            ], 500);
        }
    }


    public function applyCoupon(Request $request)
    {
        $request->validate([
            'coupon_code' => 'required|string',
            'order_total' => 'required|numeric|min:0',
        ]);

        $userId = auth()->id();
        $couponCode = $request->coupon_code;
        $orderTotal = $request->order_total;

        $coupon = Coupon::where('code', $couponCode)
            ->where('is_active', true)
            ->first();

        if (!$coupon) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid or inactive coupon code.'
            ], 404);
        }

        // Check start and end date
        $today = Carbon::today();
        if (($coupon->start_date && $today->lt(Carbon::parse($coupon->start_date))) ||
            ($coupon->end_date && $today->gt(Carbon::parse($coupon->end_date)))
        ) {
            return response()->json([
                'status' => false,
                'message' => 'Coupon is not valid at this time.'
            ], 400);
        }

        // Check minimum purchase
        if ($coupon->min_purchase && $orderTotal < $coupon->min_purchase) {
            return response()->json([
                'status' => false,
                'message' => "Minimum order of {$coupon->min_purchase} required to use this coupon."
            ], 400);
        }

        // Check max usage
        if ($coupon->max_uses && $coupon->used_count >= $coupon->max_uses) {
            return response()->json([
                'status' => false,
                'message' => 'Coupon usage limit has been reached.'
            ], 400);
        }

        // Check per-user limit
        if ($coupon->per_user_limit) {
            $userUses = \DB::table('order_discounts')
                ->join('orders', 'order_discounts.order_id', '=', 'orders.id')
                ->where('orders.user_id', $userId)
                ->where('order_discounts.coupon_code', $coupon->code)
                ->count();
            if ($userUses >= $coupon->per_user_limit) {
                return response()->json([
                    'status' => false,
                    'message' => 'You have already used this coupon the maximum allowed times.'
                ], 400);
            }
        }

        // Calculate discount
        if ($coupon->type === 'percentage') {
            $discount = ($coupon->value / 100) * $orderTotal;
        } else { // fixed
            $discount = $coupon->value;
        }

        // Ensure discount does not exceed order total
        $discount = min($discount, $orderTotal);

        return response()->json([
            'status'        => true,
            'message'       => 'Coupon applied successfully!',
            'coupon_code'   => $couponCode,
            'discount'      => $discount,
            'new_total'     => $orderTotal - $discount,
        ]);
    }


    public function trackOrder(Request $request, $orderNumber)
{
    $order = Order::with([
        'items.product.images',
        'statusHistory' => function ($q) {
            $q->orderBy('status_date', 'ASC');
        }
    ])
    ->where('order_number', $orderNumber)
    ->where('user_id', auth()->id())
    ->first();

    if (!$order) {
        return response()->json([
            'success' => false,
            'message' => 'Order not found'
        ], 404);
    }

    $firstItem = $order->items->first();
    $firstImageUrl = $firstItem?->product?->first_image_url ?? '';

    return response()->json([
        'success' => true,
        'order' => [
            'order_id'      => $order->order_number,
            'product_name'  => $firstItem?->product?->name ?? '',
            'quantity'      => $firstItem?->quantity ?? 1,
            'price'         => $firstItem?->price ?? 0,
            'image'         => $firstImageUrl,
        ],

        // 🔥 TIMELINE FIXED
        'timeline' => $order->statusHistory->map(function ($history) {
            return [
                'status'       => $this->mapStatusLabel($history->status),
                'description'  => $history->description,
                'date'         => $history->status_date->format('d M, Y'),
                'is_completed' => true
            ];
        })
    ]);
}


private function mapStatusLabel($status)
{
    return match ($status) {
        'processing'  => 'Order Confirmed',
        'shipped'     => 'Shipped',
        'in_transit'  => 'In Transit',
        'delivered'   => 'Delivered',
        'cancelled'   => 'Cancelled',
        'refund'      => 'Refunded',
        default       => ucfirst($status),
    };
}


public function myOrders(Request $request)
    {
        $orders = Order::with([
            'items.product.images'
        ])
            ->where('user_id', auth()->id())
            ->orderBy('id', 'DESC')
            ->get();

        $formatOrder = function ($order) {
            $firstItem = $order->items->first();
            $firstImageUrl = $firstItem?->product?->first_image_url ?? '';

            return [
                'order_number'   => $order->order_number,
                'order_date'     => $order->created_at->format('d M Y'),
                'order_status'   => $order->order_status,
                'product_count'  => $order->items->count(),

                'product_name'   => $firstItem?->product?->name ?? '',
                'product_image'  => $firstImageUrl,

                'items' => $order->items->map(function ($item) {
                    return [
                        'product_name'  => $item->product->name,
                        'product_image' => $item->product->first_image_url ?? '',
                        'quantity'      => $item->quantity,
                        'price'         => $item->price,
                    ];
                }),
            ];
        };

        // Group orders
        $ongoing = $orders
            ->whereIn('order_status', ['processing', 'shipped','in-transit'])
            ->values()
            ->map($formatOrder);

        $completed = $orders
            ->where('order_status', 'delivered')
            ->values()
            ->map($formatOrder);

        $cancelled = $orders
            ->whereIn('order_status', ['cancelled', 'refund'])
            ->values()
            ->map($formatOrder);

        return response()->json([
            'status' => true,
            'data' => [
                'ongoing'   => $ongoing,
                'completed' => $completed,
                'cancelled' => $cancelled,
            ]
        ]);
    }




public function storeReview(Request $request)
{
    $request->validate([
        'order_id'   => 'required|exists:orders,id',
        'product_id' => 'required|exists:products,id',
        'rating'     => 'required|integer|min:1|max:5',
        'title'      => 'nullable|string|max:255',
        'review'     => 'nullable|string',
        'photos.*'   => 'image|mimes:jpg,jpeg,png,webp|max:2048',
    ]);

    $photoNames = [];

    // ✅ SAVE REVIEW PHOTOS IN public/uploads/review
    if ($request->hasFile('photos')) {
        foreach ($request->file('photos') as $photo) {
            $filename = time() . '-' . rand(100, 999) . '.' . $photo->extension();
            $photo->move(public_path('uploads/review'), $filename);
            $photoNames[] = $filename;
        }
    }

    Review::create([
        'user_id'    => auth()->id(),
        'order_id'   => $request->order_id,
        'product_id' => $request->product_id,
        'rating'     => $request->rating,
        'title'      => $request->title,
        'review'     => $request->review,
        'photos'     => json_encode($photoNames), // store as JSON
    ]);

    return response()->json([
        'status'  => true,
        'message' => 'Review submitted successfully'
    ]);
}


}
