<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\Products;
use App\Models\Banner;
use App\Models\SubCategory;
use App\Models\Country;
use App\Models\HomeSection;
use Carbon\Carbon;

class HomeController extends Controller
{
    public function index(Request $request)
    {
        $query = Country::select('id', 'name', 'currency', 'country_code');

        if ($request->has('within_india')) {
            if ($request->within_india == 1) {
                // Only India
                $query->where('country_code', 'IN');
            } else {
                // Other countries except India
                $query->where('country_code', '!=', 'IN');
            }
        }

        return response()->json([
            'success' => true,
            'data' => $query->get()
        ]);
    }
    public function categories(Request $request)
    {
        try {
            $request->validate([
                'country_code' => 'nullable|string'
            ]);

            // 1️⃣ Get requested country
            $country = null;
            if ($request->country_code) {
                $country = Country::where('country_code', $request->country_code)->first();
            }

            // 2️⃣ Fallback to India
            if (!$country) {
                $country = Country::where('country_code', 'IN')->first();
            }

            // 3️⃣ Get categories linked to country with subcategories
            $categories = $country
                ? $country->categories()
                ->with(['subCategories' => function ($q) {
                    $q->where('is_active', 1)
                        ->select('id as subcategory_id', 'category_id', 'name', 'image');
                }])
                ->select('categories.id', 'categories.name', 'categories.image')
                ->get()
                : collect();

            return response()->json([
                'status' => true,
                'message' => 'Categories and subcategories fetched successfully',
                'data' => $categories
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    public function iconSubcategories(Request $request)
    {
        try {
            $request->validate([
                'country_code' => 'nullable|string'
            ]);

            // 1️⃣ Get country (fallback to IN)
            $country = Country::where('country_code', $request->country_code ?? 'IN')->first();

            if (!$country) {
                return response()->json([
                    'status' => true,
                    'message' => 'No data found',
                    'data' => []
                ]);
            }

            // 2️⃣ Fetch ONLY subcategories with SVG
            $subCategories = SubCategory::whereHas('category.countries', function ($q) use ($country) {
                $q->where('countries.id', $country->id);
            })
                ->where('is_active', 1)
                ->whereNotNull('svg_icon')
                ->where('svg_icon', '!=', '')
                ->select(
                    'id as subcategory_id',
                    'name',
                    'svg_icon'
                )
                ->get();

            // 3️⃣ Apply purple color to SVG
            $subCategories->transform(function ($sub) {
                $sub->svg_icon = $this->applySvgColor($sub->svg_icon, '#9b6ed9');
                return $sub;
            });

            return response()->json([
                'status' => true,
                'message' => 'Subcategories fetched successfully',
                'data' => $subCategories
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    private function applySvgColor(string $svg, string $color): string
    {
        $svg = preg_replace('/stroke="[^"]*"/', 'stroke="' . $color . '"', $svg);
        $svg = preg_replace('/fill="[^"]*"/', 'fill="' . $color . '"', $svg);

        return str_replace('<svg', '<svg style="color:' . $color . ';"', $svg);
    }


    public function home(Request $request)
    {
        $country = Country::where('country_code', $request->country_code ?? 'IN')->first();

        if (!$country) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid country'
            ], 400);
        }

        $sections = HomeSection::where('status', 1)
            ->orderBy('position')
            ->get();

        $result = [];

        foreach ($sections as $section) {

            $data = [];

            /* =====================
           ALL CATEGORY
           ===================== */
            if ($section->source_type === 'all_category') {

                // ICON STYLE → Categories

                if (in_array($section->ui_type, ['product_style', 'icon_style'])) {
                    $data = Category::select('id', 'name', 'image')
                        ->get()
                        ->map(fn($c) => [
                            'id'    => $c->id,
                            'name'  => $c->name,
                            'image' => $c->image_full
                        ]);
                }

                // PRODUCT STYLE → All products
                if ($section->ui_type === 'product_with_cart_style') {
                    $data = Products::where('status', 1)
                        ->whereHas('countries', fn($q) => $q->where('country_id', $country->id))
                        ->with('images')
                        ->take(10)
                        ->get()
                        ->map(function ($p) {

                            $price = (float) $p->price;
                            $discountAmount = (float) ($p->discount_price ?? 0);

                            $discountPercentage = $price > 0
                                ? round(($discountAmount / $price) * 100, 2)
                                : 0;

                            $finalPrice = max(0, $price - $discountAmount);

                            return [
                                'id'                  => $p->id,
                                'name'                => $p->name,
                                'original_price'               => $price,
                                'discount_price'         => round($finalPrice, 2),
                                'discount_percentage' => $discountPercentage,
                                'image'               => $p->first_image_url
                            ];
                        });
                }
            }

            /* =====================
           CATEGORY BASED
           ===================== */
            if ($section->source_type === 'category') {

                // PRODUCT STYLE
                if ($section->ui_type === 'product_with_cart_style') {
                    $data = Products::where('category_id', $section->source_id)
                        ->where('status', 1)
                        ->whereHas('countries', fn($q) => $q->where('country_id', $country->id))
                        ->with('images')
                        ->take(10)
                        ->get()
                        ->map(function ($p) {

                            $price = (float) $p->price;
                            $discountAmount = (float) ($p->discount_price ?? 0);

                            $discountPercentage = $price > 0
                                ? round(($discountAmount / $price) * 100, 2)
                                : 0;

                            $finalPrice = max(0, $price - $discountAmount);

                            return [
                                'id'                  => $p->id,
                                'name'                => $p->name,
                                'original_price'               => $price,
                                'discount_price'         => round($finalPrice, 2),
                                'discount_percentage' => $discountPercentage,
                                'image'               => $p->first_image_url
                            ];
                        });
                }

                // ICON STYLE → Subcategories


                if (in_array($section->ui_type, ['product_style', 'icon_style'])) {
                    $data = SubCategory::where('category_id', $section->source_id)
                        ->where('is_active', 1)
                        ->select('id', 'name', 'image')
                        ->get()
                        ->map(fn($s) => [
                            'id'    => $s->id,
                            'name'  => $s->name,
                            'image' => $s->image_full
                        ]);
                }
            }

            /* =====================
           SUBCATEGORY BASED
           ===================== */
            if ($section->source_type === 'subcategory') {

                $data = Products::where('sub_category_id', $section->source_id)
                    ->where('status', 1)
                    ->whereHas('countries', fn($q) => $q->where('country_id', $country->id))
                    ->with('images')
                    ->take(10)
                    ->get()
                    ->map(function ($p) {

                        $price = (float) $p->price;
                        $discountAmount = (float) ($p->discount_price ?? 0);

                        $discountPercentage = $price > 0
                            ? round(($discountAmount / $price) * 100, 2)
                            : 0;

                        $finalPrice = max(0, $price - $discountAmount);

                        return [
                            'id'                  => $p->id,
                            'name'                => $p->name,
                            'original_price'               => $price,
                            'discount_price'         => round($finalPrice, 2),
                            'discount_percentage' => $discountPercentage,
                            'image'               => $p->first_image_url
                        ];
                    });
            }

            /* =====================
           BANNER SECTION
           ===================== */


            if (in_array($section->source_type, ['banner', 'seasonal_banner'])) {

                $today = date("Y-m-d", strtotime(Carbon::today()));
                $seasonalFlag = 0;

                // 🔹 Check active seasonal banners with date range
                if ($section->source_type === 'seasonal_banner') {

                    $hasActiveSeasonal = Banner::where('status', 1)
                        ->where('seasonal_banner', 1)
                        ->whereDate('start_date', '<=', $today)
                        ->whereDate('end_date', '>=', $today)
                        ->exists();

                    $seasonalFlag = $hasActiveSeasonal ? 1 : 0;
                }

                $data = Banner::where('status', 1)
                    ->where('seasonal_banner', $seasonalFlag)

                    // 🔹 Apply date range ONLY for seasonal banners
                    ->when($seasonalFlag === 1, function ($q) use ($today) {
                        $q->whereDate('start_date', '<=', $today)
                            ->whereDate('end_date', '>=', $today);
                    })

                    ->where(function ($q) use ($country) {
                        $q->whereNull('category_id')
                            ->orWhereIn('category_id', function ($sub) use ($country) {
                                $sub->select('category_id')
                                    ->from('category_country')
                                    ->where('country_id', $country->id);
                            });
                    })
                    ->select('id', 'title', 'image', 'category_id', 'product_id')
                    ->take(10)
                    ->get()
                    ->map(fn($b) => [
                        'id' => $b->id,
                        'title' => $b->title,
                        'image' => $b->image_url,
                        'category_id' => $b->category_id,
                        'product_id' => $b->product_id,
                    ]);
            }


            $result[] = [
                'section_id' => $section->id,
                'title'      => $section->title,
                'ui_type'    => $section->ui_type,
                'source_type' => $section->source_type,
                'data'       => $data
            ];
        }

        return response()->json([
            'status'   => true,
            'country'  => $country->name,
            'sections' => $result
        ]);
    }



    public function bannerList(Request $request)
    {
        try {
            $request->validate([
                'seasonal_banner' => 'required|in:0,1'
            ]);

            $today = Carbon::today();

            $query = Banner::where('status', 1)
                ->where('seasonal_banner', $request->seasonal_banner);

            // ✅ If seasonal banner = YES
            if ($request->seasonal_banner == 1) {
                $query->whereDate('start_date', '<=', $today)
                    ->whereDate('end_date', '>=', $today);
            }

            // ✅ If seasonal banner = NO
            if ($request->seasonal_banner == 0) {
                $query->whereNull('start_date')
                    ->whereNull('end_date');
            }

            $banners = $query
                ->select('id', 'title', 'image', 'start_date', 'end_date')
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'status' => true,
                'message' => 'Banners fetched successfully',
                'data' => $banners
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    public function homeCategories(Request $request)
    {
        $request->validate([
            'country_code' => 'required|string',
            'category_id'  => 'required|integer',
            'design_style' => 'required|in:products,categories'
        ]);

        // 1️⃣ Country
        $country = Country::where('country_code', $request->country_code)->first();

        if (!$country) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid country'
            ], 404);
        }

        // 2️⃣ Validate category-country mapping
        $categoryExists = Category::where('id', $request->category_id)
            ->whereHas('countries', function ($q) use ($country) {
                $q->where('countries.id', $country->id);
            })
            ->exists();

        if (!$categoryExists) {
            return response()->json([
                'status' => false,
                'message' => 'Category not available for this country'
            ], 404);
        }

        /* =====================================
           🎯 DESIGN STYLE : CATEGORIES
        ===================================== */
        if ($request->design_style === 'categories') {

            $subCategories = SubCategory::where('category_id', $request->category_id)
                ->where('is_active', 1)
                ->select('id as subcategory_id', 'category_id', 'name', 'image')
                ->get();

            return response()->json([
                'status' => true,
                'design_style' => 'categories',
                'data' => $subCategories
            ]);
        }

        /* =====================================
           🎯 DESIGN STYLE : PRODUCTS
        ===================================== */
        /* =====================================
   🎯 DESIGN STYLE : PRODUCTS
===================================== */
        if ($request->design_style === 'products') {

            $products = Products::with('subCategory') // eager load subcategory
                ->where('category_id', $request->category_id)
                ->where('status', 1)
                ->whereHas('countries', function ($q) use ($country) {
                    $q->where('countries.id', $country->id);
                })
                ->select(
                    'id',
                    'category_id',
                    'sub_category_id',
                    'name',
                    'price',
                    'discount_price',
                    'product_type'
                )
                ->get()
                ->map(function ($product) {

                    $oldPrice = $product->price;
                    $newPrice = $product->discount_price ?? $product->price;

                    $discount = ($product->discount_price && $oldPrice > 0)
                        ? round((($oldPrice - $newPrice) / $oldPrice) * 100)
                        : 0;

                    return [
                        'product_id' => $product->id,
                        'name' => $product->name,
                        'product_type' => $product->product_type,
                        'old_price' => number_format($oldPrice, 2),
                        'new_price' => number_format($newPrice, 2),
                        'discount_percentage' => $discount,
                        'subcategory_name' => $product->subCategory?->name // ✅ add subcategory name
                    ];
                });

            return response()->json([
                'status' => true,
                'design_style' => 'products',
                'data' => $products
            ]);
        }
    }
}
