<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        Schema::table('users', function (Blueprint $table) {

            /*
            |--------------------------------------------------------------------------
            | USER ENTITY TYPE
            |--------------------------------------------------------------------------
            | Defines WHAT this user represents in system
            */
            $table->enum('user_type', [
                'super_admin',
                'cusella_internal',
                'zonal_franchise',
                'franchise',
                'dealer',
                'architect',
                'customer'
            ])->after('role_id')->index();


            /*
            |--------------------------------------------------------------------------
            | DEPARTMENT (ONLY FOR CUSELLA INTERNAL STAFF)
            |--------------------------------------------------------------------------
            */
            $table->foreignId('department_id')
                ->nullable()
                ->after('user_type')
                ->constrained('departments')
                ->nullOnDelete();


            /*
            |--------------------------------------------------------------------------
            | REPORTING HIERARCHY
            |--------------------------------------------------------------------------
            | Dealer → Franchise → Zonal → Cusella
            */
            $table->unsignedBigInteger('reporting_manager_id')
                ->nullable()
                ->after('department_id');

            $table->foreign('reporting_manager_id')
                ->references('id')
                ->on('users')
                ->nullOnDelete();


            /*
            |--------------------------------------------------------------------------
            | USER LIFECYCLE & ACCESS CONTROL
            |--------------------------------------------------------------------------
            */
            $table->date('joining_date')->nullable()->after('status');

            $table->date('access_expires_at')
                ->nullable()
                ->after('joining_date')
                ->comment('Temporary access expiry (intern/contract)');


            /*
            |--------------------------------------------------------------------------
            | LOGIN & ACTIVITY TRACKING
            |--------------------------------------------------------------------------
            */
            $table->timestamp('last_login_at')
                ->nullable()
                ->after('remember_token');
            $table->timestamp('last_activity_at')->nullable()->index();
        });
    }

    public function down(): void
    {
        Schema::table('users', function (Blueprint $table) {

            $table->dropForeign(['reporting_manager_id']);
            $table->dropForeign(['department_id']);
            $table->dropColumn([
                'user_type',
                'department_id',
                'reporting_manager_id',
                'joining_date',
                'access_expires_at',
                'last_login_at'
            ]);
        });
    }
};
