<?php

use Illuminate\Foundation\Application;
use Illuminate\Foundation\Configuration\Exceptions;
use Illuminate\Foundation\Configuration\Middleware;
use Spatie\Permission\Exceptions\UnauthorizedException;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\UnauthorizedHttpException;


return Application::configure(basePath: dirname(__DIR__))
    ->withRouting(
        web: __DIR__.'/../routes/web.php',
         api: __DIR__.'/../routes/api.php',
        commands: __DIR__.'/../routes/console.php',
        health: '/up',
    )
    ->withMiddleware(function (Middleware $middleware): void {
       $middleware->alias([
            'role' => \Spatie\Permission\Middleware\RoleMiddleware::class,
            'permission' => \Spatie\Permission\Middleware\PermissionMiddleware::class,
            'role_or_permission' => \Spatie\Permission\Middleware\RoleOrPermissionMiddleware::class,
              'checkpermission' => \App\Http\Middleware\CheckPermission::class,
        ]);
        
    // 👇 ADD THIS
    $middleware->appendToGroup('web', [
        \App\Http\Middleware\TrackUserActivity::class,
    ]);
    })
    ->withEvents(discover: true)
   ->withExceptions(function (Exceptions $exceptions): void {
        // 🔒 Catch all unauthorized access\\
      
        $exceptions->render(function (UnauthorizedException $e, $request) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Permission denied. You are not authorized to access this page.',
                ], 403);
            }

            // Redirect web users to a friendly "Access Denied" page
          //  return response()->view('permission-denied', [], 403);
          return redirect()->back()->with('error', 'You do not have permission to access this page.');
        });

        
    // ✅ Validation Errors
    $exceptions->render(function (ValidationException $e, Request $request) {
        if ($request->is('api/*')) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors'  => $e->errors(),
            ], 422);
        }
    });

    // ✅ Unauthorized (Auth / Sanctum)
    $exceptions->render(function (UnauthorizedHttpException $e, Request $request) {
        if ($request->is('api/*')) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 401);
        }
    });

    // ✅ Forbidden (Permission denied)
    $exceptions->render(function (AccessDeniedHttpException $e, Request $request) {
        if ($request->is('api/*')) {
            return response()->json([
                'success' => false,
                'message' => 'Permission denied',
            ], 403);
        }

        // Web fallback
        return redirect()->back()
            ->with('error', 'You do not have permission to access this page.');
    });

    // ✅ Model not found
    $exceptions->render(function (ModelNotFoundException $e, Request $request) {
        if ($request->is('api/*')) {
            return response()->json([
                'success' => false,
                'message' => 'Record not found',
            ], 404);
        }
    });

    // ✅ API route not found
    $exceptions->render(function (NotFoundHttpException $e, Request $request) {
        if ($request->is('api/*')) {
            return response()->json([
                'success' => false,
                'message' => 'API route not found',
            ], 404);
        }
    });

    // ✅ Fallback (500 errors)
    $exceptions->render(function (Throwable $e, Request $request) {
        if ($request->is('api/*')) {

            if (app()->environment('production')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Something went wrong',
                ], 500);
            }

            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
                'file'    => $e->getFile(),
                'line'    => $e->getLine(),
            ], 500);
        }
    });
    })
    ->create();
