<?php

namespace App\Http\Controllers;


use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\File;
use App\Models\Brands;
use Yajra\DataTables\Facades\DataTables;

class BrandController extends Controller
{
    public function addbrand(Request $request)
    {
        // Validate inputs
        $request->validate([
            'name'       => 'required|string|max:255',
            'photo'      => 'nullable|image|mimes:jpg,jpeg,png|max:4096',
        ]);

        $slug = Str::slug($request->name);
        $imagePath = null;

        if ($request->hasFile('photo')) {
            $image = $request->photo;
            $imageName = time() . '-' . rand(100, 999) . '.' . $image->extension();
            $uploadPath = public_path('assets/upload/brand/');

            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0755, true);
            }

            $image->move($uploadPath, $imageName);
            $imagePath = 'assets/upload/brand/' . $imageName;
        }

        // Create brand
        $brand = Brands::create([
            'name'  => ucfirst($request->name),
            'slug'  => $slug,
            'image' => $imagePath,
        ]);

        

        return response()->json(['success' => 'Brand added successfully!']);
    }


    public function getbrand()
    {
        $categories = Brands::orderBy('id', 'DESC')->get();

        return DataTables::of($categories)
            ->addIndexColumn()
            ->addColumn('name', fn($data) => $data->name)
         
            ->addColumn('image', function ($data) {
                return '<img src="' . asset($data->image) . '" class="avatar-md">';
            })
            ->addColumn('action', function ($row) {
                return '<a href="" class="btn btn-sm btn-purple editbrand" data-id="' . $row->id . '">Edit</a>
                    <a href="" class="btn btn-sm btn-danger deletebrand" data-id="' . $row->id . '">Delete</a>';
            })
            ->escapeColumns([])
            ->make(true);
    }


    // Delete brand (AJAX)
    public function destroy($id)
    {
        $brand = Brands::findOrFail($id);

        // Delete image from folder if exists
        if ($brand->image && File::exists(public_path($brand->image))) {
            File::delete(public_path($brand->image));
        }

        $brand->delete();
        return response()->json(['success' => 'Brand deleted successfully.']);
    }

    public function update(Request $request, $id)
{
    $request->validate([
        'name'  => 'required|string|max:100',
        'photo' => 'sometimes|file|image|mimes:jpg,jpeg,png,webp,avif|max:4096',
    ]);

    $brand = Brands::findOrFail($id);

    // Upload new image if provided
    if ($request->hasFile('photo')) {

        // delete old image
        if ($brand->image && File::exists(public_path($brand->image))) {
            File::delete(public_path($brand->image));
        }

        $image = $request->file('photo');
        $imageName = time() . '-' . rand(100, 999) . '.' . $image->extension();
        $uploadPath = public_path('assets/upload/brand/');

        if (!File::exists($uploadPath)) {
            File::makeDirectory($uploadPath, 0755, true);
        }

        $image->move($uploadPath, $imageName);
        $brand->image = 'assets/upload/brand/' . $imageName;
    }

    // Update brand details
    $brand->update([
        'name' => ucfirst($request->name),
        'slug' => Str::slug($request->name),
    ]);

    return response()->json(['success' => 'Brand updated successfully!']);
}

}
