<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Models\Wishlist;


class WishlistController extends Controller
{
    // ❤️ ADD TO WISHLIST
   public function add(Request $request)
{
    $request->validate([
        'product_id' => 'required|exists:products,id',
        'variant_id' => 'nullable|exists:product_variants,id'
    ]);

    $userId = auth()->id();

    $exists = Wishlist::where('user_id', $userId)
        ->where('product_id', $request->product_id)
        ->when($request->variant_id, function ($q) use ($request) {
            $q->where('variant_id', $request->variant_id);
        })
        ->exists();

    if ($exists) {
        return response()->json([
            'success' => false,
            'message' => 'Product already in wishlist'
        ], 409);
    }

    Wishlist::create([
        'user_id'    => $userId,
        'product_id' => $request->product_id,
        'variant_id' => $request->variant_id
    ]);

    return response()->json([
        'success' => true,
        'message' => 'Added to wishlist'
    ]);
}


    // 💔 REMOVE FROM WISHLIST
    public function remove(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'variant_id' => 'nullable|exists:product_variants,id'
        ]);

        Wishlist::where([
            'user_id'    => auth()->id(),
            'product_id' => $request->product_id,
            'variant_id' => $request->variant_id
        ])->delete();

        return response()->json([
            'success' => true,
            'message' => 'Removed from wishlist'
        ]);
    }

    // 📋 LIST WISHLIST
    public function list(Request $request)
    {
    $user = $request->user();

    $wishlists = Wishlist::where('user_id', $user->id)
        ->with([
            'product.variants' => function ($q) {
                $q->where('is_active', 1)
                  ->with([
                      'images' => function ($q) {
                          $q->where('is_primary', 1);
                      }
                  ]);
            },
            'product.reviews'
        ])
        ->get();

    $data = $wishlists->map(function ($wishlist) {

        $product = $wishlist->product;
        $variant = $product?->variants->first();
        $image   = optional($variant?->images->first())->image_path;

        return [
            'product_id'   => $product->id,
            'name'         => $product->name,
            'offer_price'  => $variant?->offer_price,
            'image'        => $image ? asset($image) : null,
            'rating'       => round($product->reviews->avg('rating') ?? 0, 1),
            'review_count' => $product->reviews->count(),
        ];
    });

    return response()->json([
        'status' => true,
        'data'   => $data
    ]);
}
}