<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\Products;
use App\Models\Banner;
use App\Models\SubCategory;
use App\Models\Product;
use App\Models\Room;

class ProductController extends Controller
{
    public function listing(Request $request)
    {
        $query = Product::where('is_active', 1)
            ->with([
                'variants' => function ($q) {
                    $q->where('is_active', 1)
                        ->with([
                            'images' => function ($q2) {
                                $q2->where('is_primary', 1);
                            }
                        ]);
                },
                'reviews'
            ]);

        // ---------------- SEARCH
        if ($request->filled('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // ---------------- CATEGORY
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        // ---------------- SUBCATEGORY
        if ($request->filled('subcategory_id')) {
            $query->where('subcategory_id', $request->subcategory_id);
        }

        // ---------------- ROOM
        if ($request->filled('room_id')) {
            $query->whereHas('rooms', function ($q) use ($request) {
                $q->where('rooms.id', $request->room_id);
            });
        }

        // ---------------- TAG
        if ($request->filled('tag_id')) {
            $query->whereHas('tags', function ($q) use ($request) {
                $q->where('tags.id', $request->tag_id);
            });
        }

        // ---------------- PRICE RANGE
        if ($request->filled('min_price') || $request->filled('max_price')) {
            $query->whereHas('variants', function ($q) use ($request) {
                if ($request->filled('min_price')) {
                    $q->where('offer_price', '>=', $request->min_price);
                }
                if ($request->filled('max_price')) {
                    $q->where('offer_price', '<=', $request->max_price);
                }
            });
        }

        // ---------------- RATING
        if ($request->filled('rating')) {
            $query->whereHas('reviews', function ($q) use ($request) {
                $q->havingRaw('AVG(rating) >= ?', [$request->rating]);
            });
        }

        // ---------------- SORT
        if ($request->filled('sort')) {
            match ($request->sort) {
                'popular'  => $query->withCount('reviews')->orderBy('reviews_count', 'desc'),
                'priority' => $query->orderBy('is_featured', 'desc'),
                default    => $query->latest(),
            };
        } else {
            $query->latest();
        }

        // ---------------- PAGINATION
        $perPage  = $request->input('per_page', 12);
        $products = $query->paginate($perPage);

        // ---------------- TRANSFORM (SAME AS productsByTag)
        $data = $products->getCollection()->map(function ($product) {

            $variant = $product->variants->first();
            $image   = optional($variant?->images->first())->image_path;

            return [
                'product_id'   => $product->id,
                'name'         => $product->name,
                'offer_price'  => $variant?->offer_price,
                'image'        => $image ? asset($image) : null,
                'rating'       => round($product->reviews->avg('rating'), 1),
                'review_count' => $product->reviews->count(),
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $data,
            'meta' => [
                'current_page' => $products->currentPage(),
                'last_page'    => $products->lastPage(),
                'per_page'     => $products->perPage(),
                'total'        => $products->total(),
            ]
        ]);
    }

    public function rooms()
    {
        $rooms = Room::select('id', 'name', 'image')
            ->orderBy('id')
            ->get()
            ->map(function ($room) {
                return [
                    'id'    => $room->id,
                    'name'  => $room->name,
                    'image' => $room->image ? asset($room->image) : null,
                ];
            });

        return response()->json([
            'success' => true,
            'data'    => $rooms
        ]);
    }

    public function categories()
    {
        $categories = Category::select(
            'id',
            'name',
            'image',
            'short_description'
        )
            ->orderBy('id')
            ->get()
            ->map(function ($category) {
                return [
                    'id'                => $category->id,
                    'name'              => $category->name,
                    'image'             => $category->image ? asset($category->image) : null,
                    'short_description' => $category->short_description,
                ];
            });

        return response()->json([
            'success' => true,
            'data'    => $categories
        ]);
    }

    public function show($id)
    {
        $product = Product::with([
            'category:id,name',
            'subCategory:id,name',
            'rooms:id,name',
            'tags:id,name',
            'variants.images',
            'variants.stocks',
            'reviews.user:id,name'
        ])
            ->where('is_active', 1)
            ->findOrFail($id);

        // ⭐ Rating calculation
        $averageRating = round($product->reviews()->avg('rating'), 1);
        $reviewCount   = $product->reviews()->count();

        /*
    |--------------------------------------------------------------------------
    | Recommendation IDs (Cross + Related + Upsell)
    |--------------------------------------------------------------------------
    */
        $recommendationIds = collect()
            ->merge($product->cross_sell_products ?? [])
            ->merge($product->related_products ?? [])
            ->merge($product->upsell_products ?? [])
            ->map(fn($id) => (int) $id)
            ->unique()
            ->filter()
            ->take(4)
            ->values();


        /*
    |--------------------------------------------------------------------------
    | Recommendation Products
    |--------------------------------------------------------------------------
    */
        $recommendations = $recommendationIds->isNotEmpty()
            ? Product::whereIn('id', $recommendationIds)
            ->where('is_active', 1)
            ->with([
                'variants' => function ($q) {
                    $q->where('is_active', 1)
                        ->with(['images' => function ($q) {
                            $q->where('is_primary', 1);
                        }]);
                },
                'reviews'
            ])
            ->get()
            ->map(function ($product) {

                $variant = $product->variants->first();
                $image   = optional($variant?->images->first())->image_path;

                return [
                    'product_id'   => $product->id,
                    'name'         => $product->name,
                    'offer_price'  => $variant?->offer_price,
                    'image'        => $image ? asset($image) : null,
                    'rating'       => round($product->reviews->avg('rating'), 1),
                    'review_count' => $product->reviews->count(),
                ];
            })
            ->values()
            : [];

        /*
    |--------------------------------------------------------------------------
    | Final Response
    |--------------------------------------------------------------------------
    */
        return response()->json([
            'success' => true,
            'data' => [
                'product' => [
                    'id'                     => $product->id,
                    'name'                   => $product->name,
                    'description'            => $product->description,
                    'material'               => $product->material,
                    'pattern'                => $product->pattern,
                    'features'               => $product->features,
                    'maintenance'            => $product->maintenance,
                    'warranty_certification' => $product->warranty_certification,
                    'measurement'            => $this->formatMeasurement($product->measurement),
                    'visibility'             => $product->visibility,
                    'gst_percent'            => $product->gst_percent,
                    'gst_inclusive'          => $product->gst_inclusive,

                    'category'    => $product->category,
                    'subcategory' => $product->subCategory,

                    'rooms' => $product->rooms->pluck('name'),
                    'tags'  => $product->tags->pluck('name'),

                    'average_rating' => $averageRating,
                    'review_count'   => $reviewCount,
                ],

                'variants' => $product->variants
                    ->where('is_active', 1)
                    ->map(function ($variant) {

                        $totalStock = $variant->stocks->sum('available_qty');

                        return [
                            'variant_id'  => $variant->id,
                            'sku'         => $variant->sku,
                            'batch_no'    => $variant->batch_no,
                            'color_name'  => $variant->color_name,
                            'color_code'  => $variant->color_code,
                            'material'    => $variant->material_or_fabric_composition,
                            'pattern'     => $variant->pattern,

                            'mrp'         => $variant->mrp,
                            'offer_price' => $variant->offer_price,
                            'unit'        => $variant->unit,
                            'moq'         => $variant->moq,

                            'dimensions'  => $variant->dimensions,

                            'images' => $variant->images->map(function ($img) {
                                return [
                                    'image'      => asset($img->image_path),
                                    'is_primary' => $img->is_primary
                                ];
                            }),

                            'stock' => [
                                'total_available_qty' => $totalStock,
                                'is_available'        => $totalStock > 0,
                            ],
                        ];
                    })
                    ->values(),

                'reviews' => $product->reviews->map(function ($review) {
                    return [
                        'user_name' => $review->user->name ?? 'Anonymous',
                        'rating'    => $review->rating,
                        'review'    => $review->review,
                        'date'      => $review->created_at->format('d M Y')
                    ];
                }),

                'recommendations' => $recommendations
            ]
        ]);
    }

    private function formatMeasurement($measurement)
    {
        if (!$measurement) {
            return null;
        }

        // Expecting "213 x 152 x 274"
        $parts = array_map('trim', explode('x', strtolower($measurement)));

        return [
            'length' => isset($parts[0]) ? (float) $parts[0] : null,
            'width'  => isset($parts[1]) ? (float) $parts[1] : null,
            'height' => isset($parts[2]) ? (float) $parts[2] : null,
        ];
    }
}
