<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Models\User;
use Illuminate\Support\Facades\Hash;
use App\Models\LoginActivity;
use App\Models\ActivityLog;
use App\Models\Location;
use App\Models\Role;
use Yajra\DataTables\DataTables;
use App\Models\BusinessProfile;
use App\Notifications\SystemNotification;
class UserController extends Controller
{
    // public function index()
    // {
    //     $users = User::with('roles')->get();
    //     $roles = Role::pluck('name');

    //     return view('admin.users.index', compact('users', 'roles'));
    // }

    public function index()
{
    $authUser = auth()->user();

    // 👑 Super Admin → all users
    if ($authUser->hasRole('Super Admin')) {
        $users = User::with('roles')->get();
    } else {

        $users = User::where(function ($q) use ($authUser) {

            // 1️⃣ Users assigned TO me
            $q->whereHas('assignedBy', function ($q2) use ($authUser) {
                $q2->where('user_assignments.user_id', $authUser->id);
            });

            // 2️⃣ OR users reporting to me
            $q->orWhere('reporting_manager_id', $authUser->id);

            // 3️⃣ OR users sharing same locations
            $q->orWhereHas('locations', function ($q3) use ($authUser) {
                $q3->whereIn(
                    'locations.id',
                    $authUser->locations()->pluck('locations.id')
                );
            });

        })
        ->with('roles')
        ->distinct()
        ->get();
    }

    $roles = Role::pluck('name');

    return view('admin.users.index', compact('users', 'roles'));
}


    public function store(Request $request)
    {
        try {
            // Validation
            $request->validate([
                'name' => 'required',
                'email' => 'required|email|unique:users',
                'password' => 'required|min:6',
                'role' => 'required',
                'user_type' => 'required',
                'status' => 'required',
            ]);

            // Create User
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'phone' => $request->phone,
                'user_type' => $request->user_type,
                'reporting_manager_id' => $request->reporting_manager_id,
                'joining_date' => $request->joining_date,
                'access_expires_at' => $request->access_expires_at,
                'status' => $request->status,
            ]);

            // Assign Role
            $role = Role::findOrFail($request->role);
            $user->assignRole([$role->id]);
         
            // Notify Admin
            $adminId = User::first()?->id;
            if ($adminId) {
                event(new \App\Events\SystemEvent([
                    'type' => 'user_created',
                    'title'        => 'New user added',
                    'subject' => $user,
                    'message' => "New user {$user->name} created",
                    'recipient_id' => $adminId,
                    'channels' => ['dashboard'],
                ]));
            }

            return response()->json([
                'status' => true,
                'message' => 'User created successfully'
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Validation errors
            return response()->json([
                'status' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            // Any other errors
            return response()->json([
                'status' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ], 500);
        }
    }


    public function edit(User $user)
    {
        return response()->json([
            'user' => $user,
            'role' => $user->roles->pluck('id')->first()
        ]);
    }

    public function update(Request $request, User $user)
    {
        try {
            $request->validate([
                'name' => 'required',
          'email' => 'required',
                'password' => 'required|min:6',
                'role' => 'required',
                'user_type' => 'required',
                'status' => 'required',
            ]);

            $user->update([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'status' => $request->status,
                'user_type' => $request->user_type,
                'reporting_manager_id' => $request->reporting_manager_id,
                'joining_date' => $request->joining_date,
                'access_expires_at' => $request->access_expires_at,
            ]);

            if ($request->password) {
                $user->update([
                    'password' => Hash::make($request->password)
                ]);
            }

            $role = Role::find($request->role);
            $user->syncRoles([$role->id]);
          

            $admin = User::first();
            $adminId = $admin?->id;
            event(new \App\Events\SystemEvent([
                'type' => 'user_updated',
                'title'        => 'User Updated',
                'subject' => $user,
                'message' => "New user {$user->name} created",
                'recipient_id' => $adminId, // Admin to notify
                'channels' => ['dashboard'],
            ]));
            return response()->json([
                'status' => true,
                'message' => 'User updated successfully'
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Validation errors
            return response()->json([
                'status' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            // Any other errors
            return response()->json([
                'status' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ], 500);
        }
    }


    public function destroy(User $user)
    {
        if ($user->id === auth()->id()) {
            return response()->json([
                'status' => false,
                'message' => 'You cannot delete yourself'
            ], 403);
        }

        $user->delete();

        return response()->json([
            'status' => true,
            'message' => 'User deleted successfully'
        ]);
    }

    public function loginHistory(User $user)
    {
        $activities = LoginActivity::where('user_id', $user->id)
            ->latest('login_at')
            ->get();

        return view('admin.users.login-history', compact('user', 'activities'));
    }

    public function deleteLoginActivity(LoginActivity $activity)
    {
        $activity->delete();

        return response()->json([
            'status' => true,
            'message' => 'Login activity deleted'
        ]);
    }
    public function deleteAllLoginHistory(User $user)
    {
        $user->loginActivities()->delete();

        return response()->json([
            'status' => true,
            'message' => 'All login history deleted'
        ]);
    }




    public function show($id)
    {
        $user = User::with(['role', 'department'])->findOrFail($id);

        $activities = ActivityLog::where('causer_id', $id)
            ->latest()
            ->limit(50)
            ->get();

        $loginLogs = LoginActivity::where('user_id', $id)
            ->latest()
            ->limit(50)
            ->get();

        return view('admin.users.profile', compact('user', 'activities', 'loginLogs'));
    }
    public function active()
    {
        return DataTables::of(
            User::where('status', 'active')
        )->make(true);
    }

    public function inactive()
    {
        return DataTables::of(
            User::where('status', 'inactive')
        )->make(true);
    }

    public function getAssignableUsers(User $user)
{
    $role = $user->roles()->first();

    if (!$role) return [];

    $departmentIds = $role->departments()->pluck('departments.id');

    return User::where('id', '!=', $user->id)
        ->whereHas('roles.departments', fn ($q) =>
            $q->whereIn('departments.id', $departmentIds)
        )
        ->select('id', 'name')
        ->get();
}
public function getAssignableLocations(User $user)
{
    // already assigned location IDs
    $assignedLocationIds = $user->locations()
        ->pluck('locations.id')
        ->toArray();

    // all locations
    $locations = Location::orderBy('name')
        ->get(['id', 'name'])
        ->map(function ($loc) use ($assignedLocationIds) {
            return [
                'id'       => $loc->id,
                'name'     => $loc->name,
                'selected' => in_array($loc->id, $assignedLocationIds),
            ];
        });

    return response()->json($locations);
}

public function saveAssignments(Request $request)
{
    $user = User::findOrFail($request->user_id);

    $user->assignedUsers()->sync($request->assigned_users ?? []);
    $user->locations()->sync($request->assigned_locations ?? []);

    return response()->json([
        'status' => true,
        'message' => 'Assignments updated'
    ]);
}
  public function changerole()
    {
        $requests = BusinessProfile::with('user')->latest()->get();
        return view('admin.business_profiles', compact('requests'));
    }

    // APPROVE
 public function approve($id)
{
    $profile = BusinessProfile::with('user')->findOrFail($id);

    // Approve profile
    $profile->update([
        'approval_status' => 'approved'
    ]);

    $user = $profile->user;

    // Determine user_type & role name
    if ($profile->business_type == 'retail_firm') {
        $userType = 'dealer';
        $roleName = 'Dealer Manager';   // change if your role name differs
    } elseif ($profile->business_type == 'construction_firm') {
        $userType = 'architect';
        $roleName = 'Architect Relations Manager'; // change if needed
    } else {
        return back()->with('error', 'Invalid Business Type');
    }

    // Get Role
    $role = Role::where('name', $roleName)->first();

    if ($role) {

        // Update user table
        $user->update([
            'user_type' => $userType,
            'role_id'   => $role->id,
        ]);

        // Update spatie role
        $user->syncRoles([$role->name]);
    }

    // ✅ Notification
    $title = 'Role Upgrade Approved';
    $message = "🎉 Your {$profile->business_type} account request has been approved successfully.";
    $eventType = 'role_approved';

    $user->notify(new SystemNotification(
        $title,
        $message,
        $eventType,
        $profile->id,
        'App\\Models\\BusinessProfile'
    ));
    return back()->with('success', 'Request Approved & User Updated Successfully');
}


    // REJECT
public function reject($id)
{
    $profile = BusinessProfile::with('user')->findOrFail($id);

    $profile->update([
        'approval_status' => 'rejected'
    ]);

    $user = $profile->user;

    if ($user) {

        $title = 'Role Upgrade Rejected';
        $message = "❌ Your {$profile->business_type} account request has been rejected. Please contact support for details.";
        $eventType = 'role_rejected';

        $user->notify(new SystemNotification(
            $title,
            $message,
            $eventType,
            $profile->id,
            'App\\Models\\BusinessProfile'
        ));
    }

    return back()->with('success', 'Request Rejected');
}


}
