<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Quotation;
use App\Models\QuotationItem;
use App\Models\QuotationItemDetail;
use App\Models\Category;
use App\Models\Project;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\QuotationArea;
use App\Models\Room;

class QuotationController extends Controller
{
    public function addToQuotation(Request $request)
    {
        DB::beginTransaction();

        try {

            $project = Project::with([
                'items.measurement',
                'items.curtain',
                'items.wallpaper',
                'items.variant',
                'items.product'
            ])->findOrFail($request->project_id);

            $quotation = Quotation::firstOrCreate(
                ['project_id' => $project->id, 'status' => 'draft'],
                [
                    'quotation_number' => 'QT-' . time(),
                    'project_name' => $project->project_name,
                    'customer_id' => $project->customer_id,
                    'customer_name' => $project->customer_name,
                    'customer_phone' => $project->customer_phone,
                    'address' => $project->address,
                    'location_lat' => $project->location_lat,
                    'location_lng' => $project->location_lng,
                    'project_type' => $project->project_type,
                    'dealer_id' => $project->dealer_id,
                    'architect_id' => $project->architect_id,
                    'status' => 'quoted',
                    'expected_completion_date' => $project->expected_completion_date,
                    'employee_name' => $project->employee_name,
                    'employee_contact' => $project->employee_contact,
                    'subtotal' => 0,
                    'tax' => 0,
                    'grand_total' => 0,
                    'created_by' => auth()->id()
                ]
            );

            // Copy project areas
            $areaMap = [];

            if ($quotation->areas()->count() == 0) {

                foreach ($project->areas as $area) {

                    $newArea = $quotation->areas()->create([
                        'room_id'    => $area->room_id,
                        'area_name'  => $area->area_name,
                        'sort_order' => $area->sort_order
                    ]);

                    // 🔥 Map project_area_id => quotation_area_id
                    $areaMap[$area->id] = $newArea->id;
                }
            } else {

                // If areas already exist, rebuild mapping
                foreach ($quotation->areas as $qArea) {

                    $projectArea = $project->areas
                        ->where('area_name', $qArea->area_name)
                        ->first();

                    if ($projectArea) {
                        $areaMap[$projectArea->id] = $qArea->id;
                    }
                }
            }

            $addedCount = 0;
            $subtotal   = $quotation->subtotal;
            $totalTax   = $quotation->tax;

            foreach ($project->items as $item) {

                if ($item->is_custom ?? 0) {
                    continue;
                }

                $exists = QuotationItem::where('quotation_id', $quotation->id)
                    ->where('project_item_id', $item->id)
                    ->exists();

                if ($exists) {
                    continue;
                }

                // 🔥 Central calculation
                $calc = $this->calculateItemAmount($item);

                if (!$calc['success']) {
                    DB::rollBack();
                    return response()->json($calc);
                }

                $price    = $calc['price'];
                $quantity = $calc['quantity'];
                $unit     = $calc['unit'];

                /* ================= GST ================= */

                $product      = $item->product;
                $gstPercent   = $product->gst_percent ?? 0;
                $gstInclusive = $product->gst_inclusive ?? 0;

                $taxAmount = 0;

                if ($gstPercent > 0) {

                    if ($gstInclusive) {
                        $base = $price / (1 + ($gstPercent / 100));
                        $taxAmount = $price - $base;
                        $price = $base;
                    } else {
                        $taxAmount = ($price * $gstPercent) / 100;
                    }
                }

                $total = $price + $taxAmount;

                $quotationItem = QuotationItem::create([
                    'quotation_id' => $quotation->id,
                    'area_id' => $areaMap[$item->area_id] ?? null,
                    'project_item_id' => $item->id,
                    'category' => $item->category,
                    'product_id' => $item->product_id,
                    'variant_id' => $item->variant_id,
                    'is_custom' => 0,
                    'quantity' => $quantity,
                    'unit' => $unit,
                    'price' => round($price, 2),
                    'total' => round($total, 2),
                    'notes' => $item->notes
                ]);

                // Save full details including new fields
                $details = [];

                if ($item->curtain) {
                    $details = $item->curtain->toArray();
                }

                if ($item->wallpaper) {
                    $details = $item->wallpaper->toArray();
                }

                QuotationItemDetail::create([
                    'quotation_item_id' => $quotationItem->id,
                    'measurement' => json_encode($item->measurement),
                    'details' => json_encode($details)
                ]);

                $subtotal += $price;
                $totalTax += $taxAmount;
                $addedCount++;
            }

            if ($addedCount == 0) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'All items already added!'
                ]);
            }

            $quotation->update([
                'subtotal'    => round($subtotal, 2),
                'tax'         => round($totalTax, 2),
                'grand_total' => round($subtotal + $totalTax, 2)
            ]);

            $project->update(['status' => 'quoted']);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => $addedCount . ' items added successfully!'
            ]);
        } catch (\Exception $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }
    private function calculateItemAmount($item)
    {
        $price = 0;
        $quantity = 0;
        $unit = null;

        /* =========================================
       🎯 CURTAIN CALCULATION
    ========================================= */
        if ($item->category == 'curtain' && $item->curtain && $item->measurement) {

            $width  = $item->measurement->width ?? 0;
            $height = $item->measurement->height ?? 0;

            $fabricWidth = $item->curtain->fabric_width ?? 120;

            $pricePerMeter =
                $item->variant->offer_price
                ?? $$item->variant->offer_price
                ?? 0;

            if ($width <= 0 || $height <= 0 || $pricePerMeter <= 0) {
                return [
                    'success' => false,
                    'message' => 'Curtain measurement or price missing for item ID ' . $item->id
                ];
            }

            $heightWithStitch = $height + 30;

            $type = strtolower($item->curtain->type ?? 'regular');

            if (in_array($type, ['eyelet', 'pleat', 'ripple', 'regular'])) {

                $panels = ceil(($width * 2.5) / $fabricWidth);
            } elseif ($type == 'roman') {

                $widthWithFold = $width + 25;
                $panels = ceil($widthWithFold / $fabricWidth);
            } else {

                $panels = ceil(($width * 2.5) / $fabricWidth);
            }

            $totalCm = $heightWithStitch * $panels;
            $totalMeters = $totalCm / 100;

            $quantity = round($totalMeters, 2);
            $price = $quantity * $pricePerMeter;
            $unit = $item->curtain->unit ?? 'meter';
        }

        /* =========================================
       🎯 WALLPAPER CALCULATION
    ========================================= */ elseif ($item->category == 'wallpaper' && $item->wallpaper) {

            $pricePerRoll = $item->variant->offer_price ?? 0;

            if ($pricePerRoll <= 0) {
                return [
                    'success' => false,
                    'message' => 'Wallpaper price missing for item ID ' . $item->id
                ];
            }

            // Manual rolls override
            if (!empty($item->wallpaper->roll_quantity)) {

                $rolls = $item->wallpaper->roll_quantity;
            } else {

                $width  = $item->measurement->width ?? 0;
                $height = $item->measurement->height ?? 0;

                $rollWidth  = $item->measurement->width ?? 0;
                $rollLength = $item->measurement->length ?? 0;
                $wallWidth  = $item->measurement->width ?? 0;
                $wallHeight = $item->measurement->height ?? 0;

                if ($rollWidth && $rollLength && $wallWidth && $wallHeight) {

                    $wallArea = $wallWidth * $wallHeight;
                    $rollArea = $rollWidth * $rollLength;

                    $rolls = ceil($wallArea / $rollArea);
                } else {

                    return [
                        'success' => false,
                        'message' => 'Wallpaper dimensions missing for item ID ' . $item->id
                    ];
                }
            }

            $quantity = $rolls;
            $price = $rolls * $pricePerRoll;
            $unit = 'roll';
        }

        /* =========================================
       🎯 NORMAL VARIANT PRODUCT
    ========================================= */ elseif ($item->variant && $item->variant->offer_price > 0) {

            $quantity = $item->quantity ?? 1;
            $price = $item->variant->offer_price * $quantity;
            $unit = $item->variant->unit;
        } else {
            return [
                'success' => false,
                'message' => 'Price configuration missing for item ID ' . $item->id
            ];
        }

        return [
            'success'  => true,
            'price'    => round($price, 2),
            'quantity' => round($quantity, 2),
            'unit'     => $unit
        ];
    }
    public function index()
    {
        return view('admin.quotations.index');
    }



    public function list(Request $request)
    {
        $quotations = Quotation::with('project')
            ->latest();

        return DataTables::of($quotations)

            ->addIndexColumn()

            ->addColumn('project_name', function ($row) {
                return $row->project->project_name ?? '-';
            })

            ->addColumn('amount', function ($row) {
                return '₹ ' . number_format($row->grand_total, 2);
            })

            ->addColumn('status_badge', function ($row) {

                $color = $row->status == 'draft' ? 'warning' : 'success';

                return '<span class="badge bg-' . $color . '">' . ucfirst($row->status) . '</span>';
            })

            ->addColumn('action', function ($row) {

                $viewUrl = route('quotations.show', $row->id);
                $deleteUrl = route('quotations.destroy', $row->id);

                return '
                <a href="' . $viewUrl . '" class="btn btn-sm btn-info">View</a>
                <button data-url="' . $deleteUrl . '" class="btn btn-sm btn-danger deleteBtn">Delete</button>
            ';
            })

            ->rawColumns(['status_badge', 'action'])
            ->make(true);
    }


    public function show($id)
    {
        $quotation = Quotation::with([
            'areas',
            'items.product',
            'items.variant',
            'items.details'
        ])->findOrFail($id);

        // ✅ correct column
        $groupedItems = $quotation->items->groupBy('area_id');

        return view('admin.quotations.show', compact('quotation', 'groupedItems'));
    }

    public function destroy($id)
    {
        $quotation = Quotation::findOrFail($id);

        $quotation->items()->delete();
        $quotation->delete();

        return response()->json([
            'success' => true,
            'message' => 'Quotation deleted successfully!'
        ]);
    }

public function update(Request $request)
{
    $quotation = Quotation::findOrFail($request->quotation_id);

    $validated = $request->validate([
        'project_name' => 'required|string|max:255',
        'customer_name' => 'required|string|max:255',
        'customer_phone' => 'nullable|string|max:255',
        'address' => 'nullable|string',
        'project_type' => 'nullable|in:home,villa,flat,commercial',
        'expected_completion_date' => 'nullable|date',
        'employee_name' => 'nullable|string|max:255',
        'employee_contact' => 'nullable|string|max:255',
        'subtotal' => 'nullable|numeric',
        'tax' => 'nullable|numeric',
        'grand_total' => 'nullable|numeric',
        'status' => 'required|string'
    ]);

    $quotation->update([
        'project_name' => $validated['project_name'],
        'customer_name' => $validated['customer_name'],
        'customer_phone' => $validated['customer_phone'] ?? null,
        'address' => $validated['address'] ?? null,
        'project_type' => $validated['project_type'] ?? null,
        'expected_completion_date' => $validated['expected_completion_date'] ?? null,
        'employee_name' => $validated['employee_name'] ?? null,
        'employee_contact' => $validated['employee_contact'] ?? null,
        'subtotal' => $validated['subtotal'] ?? 0,
        'tax' => $validated['tax'] ?? 0,
        'grand_total' => $validated['grand_total'] ?? 0,
        'status' => $validated['status'],
        'created_by' => auth()->id() // auto set logged user
    ]);

    return response()->json([
        'status' => true,
        'message' => 'Quotation updated successfully'
    ]);
}

    // public function deleteItem($id)
    // {
    //     try {

    //         $item = QuotationItem::findOrFail($id);

    //         $item->delete();

    //         return response()->json([
    //             'success' => true,
    //             'message' => 'Item deleted successfully!'
    //         ]);
    //     } catch (\Exception $e) {

    //         return response()->json([
    //             'success' => false,
    //             'message' => $e->getMessage()
    //         ]);
    //     }
    // }

    public function getItem($id)
    {
        $item = QuotationItem::with(['product', 'variant'])
            ->findOrFail($id);

        return response()->json([
            'status' => true,
            'data'   => $item
        ]);
    }


    // public function storeItem(Request $request, $roomId)
    // {
    //     DB::beginTransaction();

    //     try {

    //         $quotation = Quotation::findOrFail($request->quotation_id);

    //         /* ==============================
    //            PREPARE ITEM OBJECT FOR CALC
    //         ============================== */

    //         $fakeProjectItem = new \stdClass();
    //         $fakeProjectItem->category     = $request->category;
    //         $fakeProjectItem->product_id   = $request->product_id;
    //         $fakeProjectItem->variant_id   = $request->variant_id;
    //         $fakeProjectItem->quantity     = $request->quantity;
    //         $fakeProjectItem->measurement  = $request->measurement ?? [];
    //         $fakeProjectItem->notes        = $request->note;

    //         $fakeProjectItem->product = Product::find($request->product_id);
    //         $fakeProjectItem->variant = ProductVariant::find($request->variant_id);

    //         /* ==============================
    //            CALCULATE PRICE (REUSE LOGIC)
    //         ============================== */

    //         $calc = $this->calculateItemAmount($fakeProjectItem);

    //         if (!$calc['success']) {
    //             DB::rollBack();
    //             return response()->json($calc);
    //         }

    //         $price    = $calc['price'];
    //         $quantity = $calc['quantity'];
    //         $unit     = $calc['unit'];

    //         /* ==============================
    //            GST CALCULATION
    //         ============================== */

    //         $product      = $fakeProjectItem->product;
    //         $gstPercent   = $product->gst_percent ?? 0;
    //         $gstInclusive = $product->gst_inclusive ?? 0;

    //         $taxAmount = 0;

    //         if ($gstPercent > 0) {

    //             if ($gstInclusive) {
    //                 $base = $price / (1 + ($gstPercent / 100));
    //                 $taxAmount = $price - $base;
    //                 $price = $base;
    //             } else {
    //                 $taxAmount = ($price * $gstPercent) / 100;
    //             }
    //         }

    //         $total = $price + $taxAmount;

    //         /* ==============================
    //            CREATE QUOTATION ITEM
    //         ============================== */

    //         $quotationItem = QuotationItem::create([
    //             'quotation_id' => $quotation->id,
    //             'area_id'      => $roomId,
    //             'category'     => $request->category,
    //             'product_id'   => $request->product_id,
    //             'variant_id'   => $request->variant_id,
    //             'quantity'     => $quantity,
    //             'unit'         => $unit,
    //             'price'        => round($price, 2),
    //             'total'        => round($total, 2),
    //             'notes'        => $request->note,
    //             'is_custom'    => 1
    //         ]);

    //         /* ==============================
    //            SAVE MEASUREMENT + DETAILS
    //         ============================== */

    //         QuotationItemDetail::create([
    //             'quotation_item_id' => $quotationItem->id,
    //             'measurement' => json_encode($request->measurement ?? []),
    //             'details'     => json_encode($request->details ?? [])
    //         ]);

    //         /* ==============================
    //            UPDATE QUOTATION TOTALS
    //         ============================== */

    //         $quotation->subtotal += $price;
    //         $quotation->tax      += $taxAmount;
    //         $quotation->grand_total = $quotation->subtotal + $quotation->tax;

    //         $quotation->save();

    //         DB::commit();

    //         return response()->json([
    //             'success' => true,
    //             'message' => 'Item added successfully'
    //         ]);

    //     } catch (\Exception $e) {

    //         DB::rollBack();

    //         return response()->json([
    //             'success' => false,
    //             'message' => $e->getMessage()
    //         ]);
    //     }
    // }

    public function storeItem(Request $request, $roomId)
    {
        DB::beginTransaction();

        try {

            $quotation = Quotation::findOrFail($request->quotation_id);

            $calc = $this->prepareAndCalculate($request);

            if (!$calc['success']) {
                DB::rollBack();
                return response()->json($calc);
            }

            $quotationItem = QuotationItem::create([
                'quotation_id' => $quotation->id,
                'area_id'      => $roomId,
                'category'     => strtolower(Category::where('id', $request->category)->pluck('name')->first()),
                'product_id'   => $request->product_id,
                'variant_id'   => $request->variant_id,
                'quantity'     => $calc['quantity'],
                'unit'         => $calc['unit'],
                'price'        => $calc['price'],
                'total'        => $calc['total'],
                'notes'        => $request->note,
                'is_custom'    => 1
            ]);

            QuotationItemDetail::create([
                'quotation_item_id' => $quotationItem->id,
                'measurement' => json_encode($request->measurement ?? []),
                'details'     => json_encode($request->details ?? [])
            ]);

            $this->recalculateQuotationTotals($quotation->id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Item added successfully'
            ]);
        } catch (\Exception $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function updateItem(Request $request, $itemId)
{
    DB::beginTransaction();

    try {
        // 1️⃣ Find the quotation item
        $item = QuotationItem::findOrFail($itemId);

        // 2️⃣ Perform calculation (quantity, price, total)
        $calc = $this->prepareAndCalculate($request);
        if (!$calc['success']) {
            DB::rollBack();
            return response()->json($calc);
        }

        // 3️⃣ Update main item
        $item->update([
            'product_id' => $request->product_id,
            'variant_id' => $request->variant_id,
            'quantity'   => $calc['quantity'],
            'unit'       => $calc['unit'],
            'price'      => $calc['price'],
            'total'      => $calc['total'],
            'notes'      => $request->note,
        ]);

        // 4️⃣ Prepare measurement and details arrays dynamically
        $measurementData = [];
        $detailsData     = [];

        // Save measurement per category (curtain, wallpaper, mattress, etc.)
        foreach ($request->measurement ?? [] as $category => $values) {
            // Only save non-empty values
            $measurementData[$category] = array_filter($values, fn($v) => $v !== null && $v !== '');
        }

        // Save details per category
        foreach ($request->details ?? [] as $key => $value) {
            if (is_array($value)) {
                $detailsData[$key] = array_filter($value, fn($v) => $v !== null && $v !== '');
            } else {
                $detailsData[$key] = $value;
            }
        }

        // 5️⃣ Update or create item details (hasOne relationship)
        $details = $item->details()->first();

        if ($details) {
            $details->update([
                'measurement' => $measurementData,
                'details'     => $detailsData,
            ]);
        } else {
            $item->details()->create([
                'measurement' => $measurementData,
                'details'     => $detailsData,
            ]);
        }

        // 6️⃣ Recalculate totals for quotation
        $this->recalculateQuotationTotals($item->quotation_id);

        DB::commit();

        return response()->json([
            'success' => true,
            'message' => 'Item updated successfully'
        ]);

    } catch (\Exception $e) {
        DB::rollBack();

        return response()->json([
            'success' => false,
            'message' => $e->getMessage()
        ]);
    }
}

    // public function updateItem(Request $request, $itemId)
    // {
    //     DB::beginTransaction();

    //     try {
    //         // Find the item
    //         $item = QuotationItem::findOrFail($itemId);

    //         // Perform calculations
    //         $calc = $this->prepareAndCalculate($request);
    //         if (!$calc['success']) {
    //             DB::rollBack();
    //             return response()->json($calc);
    //         }

    //         // Update main item
    //         $item->update([
    //             'product_id' => $request->product_id,
    //             'variant_id' => $request->variant_id,
    //             'quantity'   => $calc['quantity'],
    //             'unit'       => $calc['unit'],
    //             'price'      => $calc['price'],
    //             'total'      => $calc['total'],
    //             'notes'      => $request->note,
    //         ]);

    //         // Prepare measurement/details based on filled category
    //         $measurement = [];
    //         $details     = [];

    //         if (!empty($request->measurement['curtain'])) {
    //             $measurement['curtain'] = $request->measurement['curtain'];
    //             // if (!empty($request->details['curtain'])) {
    //             //     $details['curtain'] = $request->details['curtain'];
    //             // }
    //         } elseif (!empty($request->measurement['wallpaper'])) {
    //             $measurement['wallpaper'] = $request->measurement['wallpaper'];
    //             // if (!empty($request->details['wallpaper'])) {
    //             //     $details['wallpaper'] = $request->details['wallpaper'];
    //             // }
    //         }

    //         // Save or update details relationship
    //         $details = $item->details()->first();

    //         if ($details) {
    //             print_r([
    //                 'measurement' => $measurement,
    //                 'details'     => $details,
    //             ]);
    //             $details->update([
    //                 'measurement' => $measurement,
    //                 'details'     => $details,
    //             ]);
    //         } else {
    //             $item->details()->create([
    //                 'measurement' => $measurement,
    //                 'details'     => $details,
    //             ]);
    //         }

    //         // Recalculate quotation totals
    //         $this->recalculateQuotationTotals($item->quotation_id);

    //         DB::commit();

    //         return response()->json([
    //             'success' => true,
    //             'message' => 'Item updated successfully'
    //         ]);
    //     } catch (\Exception $e) {
    //         DB::rollBack();

    //         return response()->json([
    //             'success' => false,
    //             'message' => $e->getMessage()
    //         ]);
    //     }
    // }
    // public function updateItem(Request $request, $itemId)
    // {
    //     DB::beginTransaction();

    //     try {

    //         $item = QuotationItem::findOrFail($itemId);

    //         $calc = $this->prepareAndCalculate($request);

    //         if (!$calc['success']) {
    //             DB::rollBack();
    //             return response()->json($calc);
    //         }

    //         $item->update([
    //             //  'category'   => $request->category,
    //             'product_id' => $request->product_id,
    //             'variant_id' => $request->variant_id,
    //             'quantity'   => $calc['quantity'],
    //             'unit'       => $calc['unit'],
    //             'price'      => $calc['price'],
    //             'total'      => $calc['total'],
    //             'notes'      => $request->note,
    //         ]);

    //         if ($item->details) {
    //             $item->details()->update([
    //                 'measurement' => $request->measurement ?? [],
    //                 'details'     => $request->details ?? [],
    //             ]);
    //         } else {
    //             $item->details()->create([
    //                 'measurement' => $request->measurement ?? [],
    //                 'details'     => $request->details ?? [],
    //             ]);
    //         }
    //         $this->recalculateQuotationTotals($item->quotation_id);

    //         DB::commit();

    //         return response()->json([
    //             'success' => true,
    //             'message' => 'Item updated successfully'
    //         ]);
    //     } catch (\Exception $e) {

    //         DB::rollBack();

    //         return response()->json([
    //             'success' => false,
    //             'message' => $e->getMessage()
    //         ]);
    //     }
    // }
    public function deleteItem($itemId)
    {
        DB::beginTransaction();

        try {

            $item = QuotationItem::findOrFail($itemId);
            $quotationId = $item->quotation_id;

            $item->details()->delete();
            $item->delete();

            $this->recalculateQuotationTotals($quotationId);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Item deleted successfully'
            ]);
        } catch (\Exception $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }


    private function prepareAndCalculate($request)
    {
        $fake = new \stdClass();
        $fake->category    = $request->category;
        $fake->product_id  = $request->product_id;
        $fake->variant_id  = $request->variant_id;
        $fake->quantity    = $request->quantity;
        $fake->measurement = $request->measurement ?? [];
        $fake->product     = Product::find($request->product_id);
        $fake->variant     = ProductVariant::find($request->variant_id);

        $calc = $this->calculateItemAmount($fake);

        if (!$calc['success']) {
            return $calc;
        }

        $price    = $calc['price'];
        $quantity = $calc['quantity'];
        $unit     = $calc['unit'];

        $gstPercent   = $fake->product->gst_percent ?? 0;
        $gstInclusive = $fake->product->gst_inclusive ?? 0;

        $taxAmount = 0;

        if ($gstPercent > 0) {

            if ($gstInclusive) {
                $base = $price / (1 + ($gstPercent / 100));
                $taxAmount = $price - $base;
                $price = $base;
            } else {
                $taxAmount = ($price * $gstPercent) / 100;
            }
        }

        return [
            'success'  => true,
            'price'    => round($price, 2),
            'quantity' => $quantity,
            'unit'     => $unit,
            'tax'      => round($taxAmount, 2),
            'total'    => round($price + $taxAmount, 2),
        ];
    }

    private function recalculateQuotationTotals($quotationId)
    {
        $quotation = Quotation::find($quotationId);

        $subtotal = QuotationItem::where('quotation_id', $quotationId)
            ->sum('price');

        $totalTax = QuotationItem::where('quotation_id', $quotationId)
            ->sum(DB::raw('total - price'));

        $quotation->update([
            'subtotal'    => round($subtotal, 2),
            'tax'         => round($totalTax, 2),
            'grand_total' => round($subtotal + $totalTax, 2)
        ]);
    }

  public function editItem($id)
{
    $item = QuotationItem::with(['product', 'variant', 'details'])->find($id);

    if (!$item) {
        return response()->json([
            'success' => false,
            'message' => 'Item not found'
        ]);
    }

    // Get related details
    $itemDetails = $item->details()->first();

    return response()->json([
        'success' => true,
        'data' => [
            'id' => $item->id,
            'area_id' => $item->area_id,
            'category_id' => Category::where('name', ucfirst($item->category))->pluck('id')->first(),
            'subcategory_id' => $item->product->subcategory_id ?? null,
            'product_id' => $item->product_id,
            'variant_id' => $item->variant_id,
            'quantity' => $item->quantity,
            'note' => $item->notes,
            // Access the arrays from related model
            'measurement' => $itemDetails->measurement ?? null,
            'details' => $itemDetails->details ?? null,
        ]
    ]);
}

public function updateRoom(Request $request, $roomId)
{
    $request->validate([
        'area_name' => 'required|string|max:255'
    ]);

    $room = QuotationArea::findOrFail($roomId);

    $room->update([
        'area_name' => $request->area_name
    ]);

    return response()->json([
        'success' => true,
        'message' => 'Room updated successfully'
    ]);
}

public function deleteRoom($roomId)
{
    DB::beginTransaction();

    try {

        $room = QuotationArea::findOrFail($roomId);

        // Delete all items in this room
        QuotationItem::where('area_id', $room->id)->delete();

        $quotationId = $room->quotation_id;

        $room->delete();

        // Recalculate totals
        $this->recalculateQuotationTotals($quotationId);

        DB::commit();

        return response()->json([
            'success' => true,
            'message' => 'Room deleted successfully'
        ]);

    } catch (\Exception $e) {

        DB::rollBack();

        return response()->json([
            'success' => false,
            'message' => $e->getMessage()
        ]);
    }
}

public function storeRoom(Request $request, $quotationId)
{
    $request->validate([
        'room_select' => 'required'
    ]);

    $quotation = Quotation::findOrFail($quotationId);

    if ($request->room_select === 'custom') {

        $request->validate([
            'custom_room_name' => 'required|string|max:255'
        ]);

        $areaName = $request->custom_room_name;
        $roomId = null;

    } else {

        $room = Room::findOrFail($request->room_select);

        $areaName = $room->name;
        $roomId = $room->id;
    }

    QuotationArea::create([
        'quotation_id' => $quotation->id,
        'room_id'      => $roomId,
        'area_name'    => $areaName,
        'sort_order'   => 0
    ]);

    return response()->json([
        'success' => true,
        'message' => 'Room added successfully'
    ]);
}

}
