<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Project;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\File;
use App\Models\ProjectItem;
use App\Models\QuotationItem;
use App\Models\Category;
use App\Models\SubCategory;
use App\Models\Product;
use App\Models\ProductVariant;
use Illuminate\Support\Facades\DB;

class ProjectAdminController extends Controller
{
    public function index()
    {
        $projects = Project::latest()->get();
        return view('admin.projects.index', compact('projects'));
    }

    public function show($id)
    {
        $project = Project::with([
            'areas.room',
            'areas.items.product',
            'areas.items.variant',
            'areas.items.measurement',
            'areas.items.curtain',
            'areas.items.mattress',
            'areas.items.sofa',
            'areas.items.wallpaper',
            'areas.items.accessories'
        ])->findOrFail($id);

        return view('admin.projects.show', compact('project'));
    }


    public function destroy($id)
    {
        $project = Project::findOrFail($id);
        $project->delete();

        return redirect()->route('admin.projects.index')
            ->with('success', 'Project deleted successfully');
    }



    public function data()
    {
        $projects = Project::orderBy('id', 'DESC')->get();

        return DataTables::of($projects)

            ->addIndexColumn()

            ->editColumn('expected_completion_date', function ($row) {
                return $row->expected_completion_date
                    ? \Carbon\Carbon::parse($row->expected_completion_date)->format('d M Y')
                    : '-';
            })

            ->addColumn('actions', function ($row) {

                $viewUrl = route('projects.show', $row->id);
                $deleteUrl = route('projects.destroy', $row->id);

                return '
                <a href="' . $viewUrl . '" class="btn btn-info btn-sm">View</a>

                <form action="' . $deleteUrl . '" method="POST"
                    style="display:inline-block"
                    onsubmit="return confirm(\'Are you sure?\')">
                    ' . csrf_field() . method_field('DELETE') . '
                    <button class="btn btn-danger btn-sm">Delete</button>
                </form>
            ';
            })

            ->rawColumns(['actions'])
            ->make(true);
    }


    public function uploadQuotation(Request $request, $id)
    {
        $request->validate([
            'quotation_pdf' => 'required|mimes:pdf|max:4096',
        ]);

        $project = Project::findOrFail($id);

        if ($request->hasFile('quotation_pdf')) {

            // Delete old file if exists
            if ($project->quotation_pdf && File::exists(public_path($project->quotation_pdf))) {
                File::delete(public_path($project->quotation_pdf));
            }

            $pdf = $request->file('quotation_pdf');

            $fileName = time() . '-' . str_replace(' ', '-', $project->project_name) . '.pdf';

            $uploadPath = public_path('assets/upload/project_quotation/');

            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0755, true);
            }

            $pdf->move($uploadPath, $fileName);

            $project->quotation_pdf = 'assets/upload/project_quotation/' . $fileName;
            $project->save();
        }

        return redirect()->back()->with('success', 'Quotation PDF uploaded successfully');
    }

    public function updateTotal(Request $request)
    {
        $request->validate([
            'project_id'   => 'required|exists:projects,id',
            'total_amount' => 'required|numeric|min:1'
        ]);

        $project = Project::find($request->project_id);
        $project->total_amount = $request->total_amount;
        $project->save();
        return redirect()->back()->with('success', 'Project total updated successfully');
    }



    // public function getCategoryData($category_id)
    // {
    //     $category = Category::findOrFail($category_id);

    //     // MODELS
    //     $models = $category->models
    //         ? array_map('trim', explode(',', $category->models))
    //         : [];

    //     // SUBCATEGORIES
    //     $subcategories = SubCategory::whereIn(
    //         'id',
    //         Product::where('category_id', $category_id)
    //             ->whereNotNull('subcategory_id')
    //             ->pluck('subcategory_id')
    //     )
    //         ->select('id', 'name')
    //         ->get();

    //     // ✅ CATALOGUES WITH PRODUCT INFO
    //     if ($category_id == 5) {
    //         $catalogues = Product::where('category_id', 4)
    //             ->whereNotNull('master_catalogue_name')
    //             ->select('master_catalogue_name as catalogue')
    //             ->distinct()
    //             ->orderBy('master_catalogue_name')
    //             ->get();
    //     } else {
    //         $catalogues = Product::where('category_id', $category_id)
    //             ->whereNotNull('master_catalogue_name')
    //             ->select('master_catalogue_name as catalogue')
    //             ->distinct()
    //             ->orderBy('master_catalogue_name')
    //             ->get();
    //     }


    //     return response()->json([
    //         'status' => true,
    //         'models' => $models,
    //         'subcategories' => $subcategories,
    //         'catalogues' => $catalogues
    //     ]);
    // }

    public function getCategoryData($category_id)
    {
        $category = Category::findOrFail($category_id);

        // MODELS
        $models = $category->models
            ? array_map('trim', explode(',', $category->models))
            : [];

        // SUBCATEGORIES
        $subcategories = SubCategory::whereIn(
            'id',
            Product::where('category_id', $category_id)
                ->whereNotNull('subcategory_id')
                ->pluck('subcategory_id')
        )
            ->select('id', 'name')
            ->get();

        /*
    |--------------------------------------------------------------------------
    | CATALOGUES
    |--------------------------------------------------------------------------
    */

        if ($category->slug === 'mattress') {
            $catId = ($category_id == 5) ? 4 : $category_id;
            // ❌ NO DISTINCT for mattress
            $catalogues = Product::where('category_id', $category_id)
                ->whereNotNull('master_catalogue_name')
                ->select(
                    'id as product_id',
                    'name as product_name',
                    'master_catalogue_name as catalogue'
                )
                ->get();
        } elseif ($category->slug === 'curtain') {
            $catId = ($category_id == 5) ? 4 : $category_id;

            // ✅ DISTINCT catalogue for other categories
            $catalogues = Product::where('category_id', $catId)
                ->whereNotNull('master_catalogue_name')
                ->selectRaw('
                MIN(id) as product_id,
                MIN(name) as product_name,
                master_catalogue_name as catalogue
            ')
                ->groupBy('master_catalogue_name')
                ->get();
        } else {
            $catalogues = Product::where('category_id', $category_id)
                ->whereNotNull('master_catalogue_name')
                ->selectRaw('
                MIN(id) as product_id,
                MIN(name) as product_name,
                master_catalogue_name as catalogue
            ')
                ->groupBy('master_catalogue_name')
                ->get();
        }

        return response()->json([
            'status' => true,
            'models' => $models,
            'subcategories' => $subcategories,
            'catalogues' => $catalogues
        ]);
    }

    //SUBCATEGORY CHANGE
    // public function getSubcategoryCatalogues($subcategory_id)
    // {
    //     // Get subcategory
    //     $subcategory = SubCategory::findOrFail($subcategory_id);
    //     $models = $subcategory->models
    //         ? array_map('trim', explode(',', $subcategory->models))
    //         : [];
    //     // Get its category
    //     $category_id = $subcategory->category_id;

    //     // If Curtain category (5), use Fabric category (4)
    //     $catalogueCategoryId = ($category_id == 5) ? 4 : $category_id;

    //     $catalogues = Product::where('category_id', $catalogueCategoryId)
    //         ->whereNotNull('master_catalogue_name')
    //         ->select(
    //             'id as product_id',
    //             'name as product_name',
    //             DB::raw("master_catalogue_name as catalogue")
    //         )
    //         ->distinct()
    //         ->get();

    //     return response()->json([
    //         'status' => true,
    //         'models' => $models,
    //         'catalogues' => $catalogues
    //     ]);
    // }

public function getSubcategoryCatalogues($subcategory_id)
{
    $subcategory = SubCategory::findOrFail($subcategory_id);
    $category = Category::findOrFail($subcategory->category_id);

    $models = $subcategory->models
        ? array_map('trim', explode(',', $subcategory->models))
        : [];

    if ($category->slug === 'mattress') {

        $catalogues = Product::where('category_id', $category->id)
            ->where('subcategory_id', $subcategory_id)
            ->whereNotNull('master_catalogue_name')
            ->select(
                'id as product_id',
                'name as product_name',
                'master_catalogue_name as catalogue'
            )
            ->get();

    } else {

        $catalogueCategoryId = ($category->slug === 'curtain') ? 4 : $category->id;

        $catalogues = Product::where('category_id', $catalogueCategoryId)
            ->whereNotNull('master_catalogue_name')
            ->selectRaw('
                MIN(id) as product_id,
                MIN(name) as product_name,
                master_catalogue_name as catalogue
            ')
            ->groupBy('master_catalogue_name')
            ->orderBy('master_catalogue_name')
            ->get();
    }

    return response()->json([
        'status' => true,
        'models' => $models,
        'catalogues' => $catalogues
    ]);
}


    //CATALOGUE SELECTED → LOAD SKU
    public function getCatalogueSkus(Request $request)
    {
        $product_id = $request->product_id;

        // 1️⃣ Get product
        $product = Product::find($product_id);

        if (!$product) {
            return response()->json([
                'status' => false,
                'message' => 'Product not found'
            ]);
        }

        // 2️⃣ Get master catalogue name
        $masterCatalogueName = $product->master_catalogue_name;

        // 3️⃣ Get all product IDs having same master_catalogue_name
        $productIds = Product::where('master_catalogue_name', $masterCatalogueName)
            ->pluck('id');

        // 4️⃣ Get all variants of those products
        $variants = ProductVariant::whereIn('product_id', $productIds)->get();


        $skus = $variants->map(function ($v) {

            // ✅ SOFA DIMENSION FORMAT
            $sofa_dimension_label = null;

            if ($v->dimensions) {

                $dim = is_string($v->dimensions)
                    ? json_decode($v->dimensions, true)
                    : $v->dimensions;

                if (is_array($dim)) {

                    $length = $dim['length'] ?? null;
                    $width  = $dim['width'] ?? null;
                    $height = $dim['height'] ?? null;

                    // convert cm → inch if needed
                    if ($v->unit == 'cm') {
                        $length = $length ? round($length / 2.54) : null;
                        $width  = $width  ? round($width / 2.54)  : null;
                        $height = $height ? round($height / 2.54) : null;
                    }

                    if ($length && $width && $height) {
                        $sofa_dimension_label =
                            "L {$length} x W {$width} x H {$height}";
                    }
                }
            }

            return [
                'variant_id' => $v->id,
                'product_id' => $v->product_id,
                'sku' => $v->sku,
                // 'batch_no' => $v->batch_no,
                'mrp' => $v->mrp,
                'offer_price' => $v->offer_price,
                'unit' => $v->unit,
                // 'color_name' => $v->color_name,
                // 'color_code' => $v->color_code,
                // 'pattern' => $v->pattern,
                // 'material_or_fabric_composition' => $v->material_or_fabric_composition,
                // 'gsm' => $v->gsm,
                // 'weight' => $v->weight,
                // 'rub_count' => $v->rub_count,
                // 'moq' => $v->moq,
                // 'is_active' => $v->is_active,
                // 'fast_moving' => $v->fast_moving,
                // 'slow_moving' => $v->slow_moving,
                // 'design_repeat' => $v->design_repeat,
                // 'mattress_size_category' => $v->mattress_size_category,
                // 'mattress_length_inch' => $v->mattress_length_inch,
                // 'mattress_width_inch' => $v->mattress_width_inch,
                // 'mattress_thickness_inch' => $v->mattress_thickness_inch,
                'dimensions' => $sofa_dimension_label
            ];
        });
        // ✅ Mattress Config
        $mattress_config = [
            'categories' => [],
            'thickness'  => [],
            'units'      => ['inch', 'ft', 'cm'],
            'sizes'      => []
        ];

        foreach ($variants as $v) {

            if (
                !$v->mattress_size_category ||
                !$v->mattress_length_inch ||
                !$v->mattress_width_inch
            ) {
                continue;
            }

            $category = strtolower($v->mattress_size_category);

            // categories
            if (!in_array($category, $mattress_config['categories'])) {
                $mattress_config['categories'][] = $category;
            }

            // thickness
            if (
                $v->mattress_thickness_inch &&
                !in_array($v->mattress_thickness_inch, $mattress_config['thickness'])
            ) {
                $mattress_config['thickness'][] = $v->mattress_thickness_inch;
            }

            // inch
            $inch_length = $v->mattress_length_inch;
            $inch_width  = $v->mattress_width_inch;

            // ft
            $ft_length = round($inch_length / 12, 2);
            $ft_width  = round($inch_width / 12, 2);

            // cm
            $cm_length = round($inch_length * 2.54);
            $cm_width  = round($inch_width * 2.54);

            $mattress_config['sizes'][$category][] = [
                'variant_id' => $v->id,
                'inch' => "{$inch_length}x{$inch_width}",
                'ft'   => "{$ft_length}x{$ft_width}",
                'cm'   => "{$cm_length}x{$cm_width}",
            ];
        }


        return response()->json([
            'status' => true,
            'skus' => $skus,
            'mattress_config' => $mattress_config,

        ]);
    }



    public function getProjectItems(Request $request)
    {
        $itemId = $request->item_id;

        $items = ProjectItem::with([
            'measurement',
            'curtain',
            'mattress',
            'wallpaper',
            'product',
            'variant'
        ])
            ->where('id', $itemId)
            ->get();

        $data = $items->map(function ($item) {

            return [
                'id'          => $item->id,
                'project_id'  => $item->project_id,
                'area_id'     => $item->area_id,
                'category'    => $item->category,
                'product_id'  => $item->product_id,
                'variant_id'  => $item->variant_id,
                'product_name' => optional($item->product)->name,
                'variant_sku' => optional($item->variant)->sku,
                'quantity'    => $item->quantity,
                'notes'       => $item->notes,

                'measurement' => $item->measurement,

                'details' => match ($item->category) {
                    'curtain'   => $item->curtain,
                    'mattress'  => $item->mattress,
                    'wallpaper' => $item->wallpaper,
                    default     => null
                }
            ];
        });

        return response()->json([
            'success' => true,
            'data'    => $data
        ]);
    }
    // public function addToQuotation(Request $request)
    // {
    //     $request->validate([
    //         'project_item_id' => 'required|exists:project_items,id'
    //     ]);

    //     $item = ProjectItem::findOrFail($request->project_item_id);

    //     QuotationItem::updateOrCreate(
    //         [
    //             'project_item_id' => $item->id
    //         ],
    //         [
    //             'project_id' => $item->project_id
    //         ]
    //     );

    //     return back()->with('success', 'Added to quotation list');
    // }

    public function update(Request $request)
    {
        try {

            $project = Project::findOrFail($request->project_id);

            $data = $request->validate([
                'project_name' => 'required|string',
                'customer_name' => 'required|string',
                'customer_phone' => 'nullable|string',
                'address' => 'nullable|string',
                'project_type' => 'nullable|string',
                'status' => 'required|string',
                'expected_completion_date' => 'nullable|date',
                'employee_name' => 'nullable|string',
                'employee_contact' => 'nullable|string',
                'total_amount' => 'nullable|numeric'
            ]);

            $project->update($data);

            return response()->json([
                'status' => true,
                'message' => 'Project updated successfully',
                'project' => $project
            ]);
        } catch (\Exception $e) {

            return response()->json([
                'status' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
