<?php


namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\ProductVariant;
use App\Models\ProductImages;
use App\Models\Stock;
use Illuminate\Support\Str;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use App\Events\SystemEvent;
use App\Models\User;
use App\Models\SubCategory;

class ProductController extends Controller
{
    public function storeStepOne(Request $request)
    {
        $validated = $request->validate([
            'category_id'     => 'required|exists:categories,id',
            'sub_category_id' => 'nullable|exists:sub_categories,id',
            'name'            => 'required|string|max:855',
            'description'     => 'nullable|string',

            'length'          => 'nullable|numeric',
            'width'           => 'nullable|numeric',
            'height'          => 'nullable|numeric',

            'features'        => 'nullable|string|max:855',
            'maintenance'     => 'nullable|string|max:855',
            'warranty_certification' => 'nullable|string|max:855',

            'rooms'           => 'nullable|array',
            'rooms.*'         => 'exists:rooms,id',
            'tags'            => 'nullable|array',
            'tags.*'          => 'exists:tags,id',

            'master_catalogue_name' => 'required|string|max:255',
            'master_catalogue_code' => 'required|string|max:100',

            // Extra fields
            'visibility'          => 'required|in:B2C,B2B,Franchise,All',
            'gst_percent'         => 'nullable|numeric|min:0|max:100',
            'gst_inclusive'       => 'required|boolean',
            'is_featured'         => 'required|boolean',
            'related_products'    => 'nullable|array',
            'related_products.*'  => 'exists:products,id',
            'cross_sell_products' => 'nullable|array',
            'cross_sell_products.*' => 'exists:products,id',
            'upsell_products'     => 'nullable|array',
            'upsell_products.*'   => 'exists:products,id',
            'seo_title'           => 'nullable|string|max:255',
            'seo_description'     => 'nullable|string',
            'search_keywords'     => 'nullable|string',
            'care_instructions'   => 'nullable|string',
            'fast_moving'         => 'nullable|boolean',
            'slow_moving'         => 'nullable|boolean',
            'sample_purchase'    => 'required|boolean',
            'sample_purchase_rate'    => 'required|numeric',
        ]);

        // Calculate measurement
        $measurement = null;

        if ($request->filled('length') || $request->filled('width') || $request->filled('height')) {

            $length = $request->length ?? '-';
            $width  = $request->width ?? '-';
            $height = $request->height ?? '-';

            $measurement = "{$length} x {$width} x {$height}";
        }


        DB::beginTransaction();
        try {
            // Map sub_category_id to subcategory_id in DB
            $productData = $validated;
            $productData['subcategory_id'] = $validated['sub_category_id'];
            unset($productData['sub_category_id']);

            // Add calculated measurement
            $productData['measurement'] = $measurement;

            // Create product
            $product = Product::create($productData);

            // Attach rooms to product_rooms table
            if ($request->filled('rooms')) {
                $product->rooms()->sync($request->rooms);
            }

            // Attach tags to product_tags table
            if ($request->filled('tags')) {
                $product->tags()->sync($request->tags);
            }

            DB::commit();


            //*** notification ********/
            $adminId = User::first()?->id;

            event(new SystemEvent([
                'event_type'   => 'product_created',
                'subject'      => $product,
                'message'      => "New product " . $product->name . " added",
                'recipient_id' => User::first()?->id,
                'channels'     => ['dashboard'],
                'changes'      => $product->toArray(),
            ]));

            //*** notification end********/
            return response()->json([
                'success' => true,
                'product_id' => $product->id,
                'message' => 'Step 1 saved successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong. ' . $e,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function storeStepTwo(Request $request)

    {
        try {

            $data = $request->validate([
                'product_id' => 'required|exists:products,id',
                'sku' => 'required',
                'mrp' => 'required|numeric',
                'offer_price' => 'required|numeric',
                'unit' => 'required',
            ]);

            $product = Product::with('category')->findOrFail($data['product_id']);

            // ---------- DIMENSIONS ----------
            $dimensions = [
                'length' => $request->input('dimensions.length'),
                'width'  => $request->input('dimensions.width'),
                'height' => $request->input('dimensions.height'),
            ];

            if (empty(array_filter($dimensions))) {
                $dimensions = null;
            }

            // ---------- MATTRESS ----------
            $mattressData = [];

            if ($product->category->name == 'Mattress') {

                $mattressData = [
                    'mattress_size_category'   => $request->mattress_size_category[0] ?? null,
                    'mattress_length_inch'    => $request->mattress_length_inch[0] ?? null,
                    'mattress_width_inch'     => $request->mattress_width_inch[0] ?? null,
                    'mattress_thickness_inch' => $request->mattress_thickness_inch[0] ?? null,
                ];
            }

            // ---------- CREATE VARIANT ----------
            $variant = ProductVariant::create([
                'product_id' => $data['product_id'],
                'is_default'=> $request->is_default,
                'sku' => $request->sku,
                'batch_no' => $request->batch_no,
                'mrp' => $request->mrp,
                'offer_price' => $request->offer_price,
                'unit' => $request->unit,
                'dimensions' => $dimensions,

                'color_name' => $request->color_name,
                'color_code' => $request->color_code,
                'material_or_fabric_composition' => $request->material_or_fabric_composition,
                'gsm' => $request->gsm,
                'weight' => $request->weight,
                'rub_count' => $request->rub_count,
                'moq' => $request->moq,
                'pattern' => $request->pattern,
                'design_repeat' => $request->design_repeat,
            ] + $mattressData);

            return response()->json([
                'success' => true,
                'message' => 'Variant saved successfully',
                'variant' => $variant
            ]);
        } catch (\Throwable $e) {

            return response()->json([
                'success' => false,
                'message' => 'Variant save failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    public function variantForm(Request $request)
    {
        return view('admin.product.variant-form', [
            'variant' => new ProductVariant(),
            'index' => $request->index,
            'product' => Product::find(request('product_id')) ?? null
        ]);
    }

    // public function storeImages(Request $request)
    // {
    //     $request->validate([
    //         'product_variant_id' => 'required|exists:product_variants,id',
    //         'images.*' => 'required|image|max:2048',
    //         'primary_image' => 'required|integer'
    //     ]);

    //     if ($request->hasFile('images')) {
    //         foreach ($request->file('images') as $file) {

    //             $name = time() . '-' . rand(100, 999) . '.' . $file->extension();

    //             // Destination path
    //             $destination = public_path('assets/upload/products');

    //             // Move file
    //             $file->move($destination, $name);

    //             ProductImages::create([
    //                 'product_variant_id' => $request->product_variant_id,
    //                 // store FULL RELATIVE PATH
    //                 'image_path' => 'assets/upload/products/' . $name,
    //                 'type' => 'front'
    //             ]);
    //         }
    //     }


    //     // foreach ($request->file('images') as $image) {
    //     //     $path = $image->store('products', 'public');

    //     //     ProductImages::create([
    //     //         'product_variant_id' => $request->product_variant_id,
    //     //         'image_path' => $path,
    //     //         'type' => 'front'
    //     //     ]);
    //     // }

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Images uploaded successfully'
    //     ]);
    // }



    public function storeStock(Request $request)
    {
        $validated = $request->validate([
            'stocks' => 'required|array|min:1',

            'stocks.*.product_variant_id' => 'required|exists:product_variants,id',
            'stocks.*.location_id'        => 'required|exists:locations,id',

            'stocks.*.available_qty' => 'required|numeric|min:0',
            'stocks.*.reorder_level' => 'nullable|numeric|min:0',

            'stocks.*.opening_qty'   => 'nullable|numeric|min:0',
            'stocks.*.received_qty'  => 'nullable|numeric|min:0',
            'stocks.*.damaged_qty'   => 'nullable|numeric|min:0',
            'stocks.*.returned_qty'  => 'nullable|numeric|min:0',
        ]);

        foreach ($validated['stocks'] as $stock) {

            Stock::updateOrCreate(
                [
                    'product_variant_id' => $stock['product_variant_id'],
                    'location_id'        => $stock['location_id'],
                ],
                [
                    'available_qty' => $stock['available_qty'],
                    'reorder_level' => $stock['reorder_level'] ?? null,

                    'opening_qty'   => $stock['opening_qty']  ?? 0,
                    'received_qty'  => $stock['received_qty'] ?? 0,
                    'damaged_qty'   => $stock['damaged_qty']  ?? 0,
                    'returned_qty'  => $stock['returned_qty'] ?? 0,

                    'is_available'  => $stock['available_qty'] > 0,
                ]
            );
        }

        return response()->json([
            'success' => true,
            'message' => 'Stock saved successfully',
        ]);
    }



    public function index()
    {

        return view('admin.product.product');
    }



    public function getProducts(Request $request)
    {
       $products = Product::with([
        'category',
        'subCategory',
        'variants.images',
        'variants.stocks',
        'tags',
        'rooms'
    ]);

    // Filter by category
    if ($request->category) {
        $products->where('category_id', $request->category);
    }

    // Filter by subcategory
    if ($request->subcategory) {
        $products->where('subcategory_id', $request->subcategory);
    }

    // Filter by tag (many-to-many)
    if ($request->tag) {
        $products->whereHas('tags', function($q) use ($request) {
            $q->where('tags.id', $request->tag);
        });
    }

    // Filter by room (many-to-many)
    if ($request->room) {
        $products->whereHas('rooms', function($q) use ($request) {
            $q->where('rooms.id', $request->room);
        });
    }

    // Filter by price range (on variant offer_price)
    if ($request->min_price || $request->max_price) {
        $products->whereHas('variants', function($q) use ($request) {

            if ($request->min_price) {
                $q->where('offer_price', '>=', $request->min_price);
            }

            if ($request->max_price) {
                $q->where('offer_price', '<=', $request->max_price);
            }

        });
    }

    $products = $products->latest();


        return DataTables::of($products)
            ->addIndexColumn()

            ->addColumn('category', fn($p) => $p->category->name ?? '-')

            ->addColumn('subcategory', fn($p) => $p->subCategory->name ?? '-')
            ->addColumn('images', function ($p) {

                $images = $p->variants->flatMap(fn($v) => $v->images)->take(3);

                if ($images->isEmpty()) {
                    return '-';
                }

                $html = '<div class="d-flex align-items-center gap-1 flex-nowrap">';

                foreach ($images as $i) {

                    $path = is_array($i->image_path)
                        ? implode('/', $i->image_path)
                        : $i->image_path;

                    $url = asset($path);

                    $html .= '<img src="' . $url . '" height="40"
                    class="rounded"
                    style="width:40px;object-fit:cover;">';
                }

                $html .= '</div>';

                return $html;
            })




            ->addColumn('price', function ($p) {
                $variant = $p->variants->first();
                return $variant
                    ? '₹' . $variant->offer_price
                    : '-';
            })

            ->addColumn('stock', function ($p) {
                return $p->variants->sum(
                    fn($v) =>
                    $v->stocks->sum('available_qty')
                );
            })

            ->addColumn('status', function ($p) {
                return $p->is_active
                    ? '<span class="badge bg-success">Active</span>'
                    : '<span class="badge bg-danger">Inactive</span>';
            })

            ->addColumn('action', function ($p) {
                return '
                <a href="' . route('edit-product', ['id' => $p->id]) . '"
                   class="btn btn-sm btn-info">Edit</a>

                <button class="btn btn-sm btn-danger deleteProduct"
                    data-id="' . $p->id . '">
                    Delete
                </button>
            ';
            })

            ->rawColumns(['images', 'status', 'action'])
            ->make(true);
    }
// public function getProducts(Request $request)
// {
//     $products = Product::with([
//         'category',
//         'subCategory',
//         'variants.images',
//         'variants.stocks',
//         'tags',
//         'rooms'
//     ]);

//     // Filter by category
//     if ($request->category) {
//         $products->where('category_id', $request->category);
//     }

//     // Filter by subcategory
//     if ($request->subcategory) {
//         $products->where('sub_category_id', $request->subcategory);
//     }

//     // Filter by tag (many-to-many)
//     if ($request->tag) {
//         $products->whereHas('tags', function($q) use ($request) {
//             $q->where('tags.id', $request->tag);
//         });
//     }

//     // Filter by room (many-to-many)
//     if ($request->room) {
//         $products->whereHas('rooms', function($q) use ($request) {
//             $q->where('rooms.id', $request->room);
//         });
//     }

//     // Filter by price range (on variant offer_price)
//     if ($request->min_price || $request->max_price) {
//         $products->whereHas('variants', function($q) use ($request) {

//             if ($request->min_price) {
//                 $q->where('offer_price', '>=', $request->min_price);
//             }

//             if ($request->max_price) {
//                 $q->where('offer_price', '<=', $request->max_price);
//             }

//         });
//     }

//     $products = $products->latest();

//     return DataTables::of($products)
//         ->addIndexColumn()
//         ->addColumn('category', fn($p) => $p->category->name ?? '-')
//         ->addColumn('subcategory', fn($p) => $p->subCategory->name ?? '-')
//         ->make(true);
// }

public function getSubcategories($categoryId)
{
    return SubCategory::where('category_id', $categoryId)->get();
}


    public function destroy($id)
    {
        DB::beginTransaction();

        try {
            $product = Product::with([
                'variants.images',
                'variants.stocks'
            ])->findOrFail($id);

            // 1️⃣ Delete variant images (FILES + DB)
            foreach ($product->variants as $variant) {
                foreach ($variant->images as $image) {

                    // Full physical file path
                    $filePath = public_path($image->image_path);

                    if (file_exists($filePath)) {
                        unlink($filePath);
                    }

                    $image->delete();
                }

                // 2️⃣ Delete stocks
                $variant->stocks()->delete();

                // 3️⃣ Delete variant
                $variant->delete();
            }

            // 4️⃣ Detach relations
            $product->rooms()->detach();
            $product->tags()->detach();

            // 5️⃣ Delete product
            $product->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Product deleted successfully'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Delete failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    public function show($id)
    {
        $product = Product::with([
            'category',
            'subcategory',
            'variants.images',
            'variants.stocks'
        ])->findOrFail($id);

        //     $variant = $product->variants->first();
        // print_r($variant?->images);die();

        return view('admin.product.edit-product', compact('product'));
    }





    public function stepOneUpdate(Request $request)
    {
        // 1️⃣ Validation
        $request->validate([
            'id' => 'required|exists:products,id',
            'category_id' => 'required|exists:categories,id',
            'sub_category_id' => 'nullable|exists:sub_categories,id',
            'name' => 'required|string|max:255',
            'features' => 'nullable|string',
            'warranty_certification' => 'nullable|string',
            'maintenance' => 'nullable|string',

            // Measurement
            'length' => 'nullable|numeric',
            'width'  => 'nullable|numeric',
            'height' => 'nullable|numeric',

            'description' => 'nullable|string',
            'rooms' => 'nullable|array',
            'rooms.*' => 'exists:rooms,id',
            'tags' => 'nullable|array',
            'tags.*' => 'exists:tags,id',
            'master_catalogue_name' => 'required|string|max:255',
            'master_catalogue_code' => 'required|string|max:100',

            // ✅ Extra fields validation
            'visibility'          => 'required|in:B2C,B2B,Franchise,All',
            'gst_percent'         => 'nullable|numeric|min:0|max:100',
            'gst_inclusive'       => 'required|boolean',
            'is_featured'         => 'required|boolean',
            'related_products'    => 'nullable|array',
            'related_products.*'  => 'exists:products,id',
            'cross_sell_products' => 'nullable|array',
            'cross_sell_products.*' => 'exists:products,id',
            'upsell_products'     => 'nullable|array',
            'upsell_products.*'   => 'exists:products,id',
            'seo_title'           => 'nullable|string|max:255',
            'seo_description'     => 'nullable|string',
            'search_keywords'     => 'nullable|string',
            'care_instructions'   => 'nullable|string',
            'fast_moving'         => 'nullable|boolean',
            'slow_moving'         => 'nullable|boolean',
            'sample_purchase'    => 'nullable|boolean',
        ]);

        DB::beginTransaction();

        try {
            $product = Product::findOrFail($request->id);

            // 🔹 Build measurement string
            $measurement = null;

            if ($request->filled('length') || $request->filled('width') || $request->filled('height')) {

                $length = $request->length ?? '-';
                $width  = $request->width ?? '-';
                $height = $request->height ?? '-';

                $measurement = "{$length} x {$width} x {$height}";
            }


            // 2️⃣ Update product table (existing + extra fields)
            $product->update([
                'category_id' => $request->category_id,
                'subcategory_id' => $request->sub_category_id,
                'master_catalogue_name' => $request->master_catalogue_name,
                'master_catalogue_code' => $request->master_catalogue_code,
                'name' => $request->name,
                'features' => $request->features,
                'warranty_certification' => $request->warranty_certification,
                'maintenance' => $request->maintenance,
                'measurement' => $measurement,
                'description' => $request->description,

                // ✅ Extra fields
                'visibility' => $request->visibility,
                'gst_percent' => $request->gst_percent,
                'gst_inclusive' => $request->gst_inclusive,
                'is_featured' => $request->is_featured,
                'related_products' => $request->related_products,
                'cross_sell_products' => $request->cross_sell_products,
                'upsell_products' => $request->upsell_products,
                'seo_title' => $request->seo_title,
                'seo_description' => $request->seo_description,
                'search_keywords' => $request->search_keywords,
                'care_instructions' => $request->care_instructions,
                'fast_moving' => $request->fast_moving,
                'slow_moving' => $request->slow_moving,
                'sample_purchase'    => $request->sample_purchase,
                'sample_purchase_rate'    => $request->sample_purchase_rate,
            ]);

            // 3️⃣ Sync rooms & tags
            $product->rooms()->sync($request->rooms);
            $product->tags()->sync($request->tags ?? []);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Product details updated successfully',
                'product_id' => $product->id
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Update failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    public function stepTwoUpdate(Request $request)
    {
        $variantId = $request->variant_id;

        $request->validate([
            'product_id'   => ['required', 'exists:products,id'],
            'sku'          => [
                'required',
                'string',
                'max:255',
                Rule::unique('product_variants')
                    ->where(fn($q) => $q->where('product_id', $request->product_id))
                    ->ignore($variantId),
            ],
            'unit'         => 'required|string',
            'mrp'          => 'required|numeric',
            'offer_price'  => 'required|numeric',
        ]);

        DB::beginTransaction();

        try {

            // ---------- DIMENSIONS ----------
            $dimensions = null;

            if ($request->dimensions) {

                $filtered = array_filter($request->dimensions);

                if (!empty($filtered)) {
                    $dimensions = $request->dimensions;
                }
            }

            // ---------- MATTRESS ----------
            $mattressSize   = is_array($request->mattress_size_category)
                ? $request->mattress_size_category[0] ?? null
                : $request->mattress_size_category;

            $mattressLength = is_array($request->mattress_length_inch)
                ? $request->mattress_length_inch[0] ?? null
                : $request->mattress_length_inch;

            $mattressWidth  = is_array($request->mattress_width_inch)
                ? $request->mattress_width_inch[0] ?? null
                : $request->mattress_width_inch;

            $mattressThick  = is_array($request->mattress_thickness_inch)
                ? $request->mattress_thickness_inch[0] ?? null
                : $request->mattress_thickness_inch;

            // ---------- UPDATE ----------
            $variant = ProductVariant::updateOrCreate(
                [
                    'id' => $variantId,
                    'product_id' => $request->product_id,
                ],
                [
                    'is_default'=> $request->is_default,
                    'sku'        => $request->sku,
                    'batch_no'   => $request->batch_no,
                    'unit'       => $request->unit,
                    'dimensions' => $dimensions,

                    'material_or_fabric_composition' => $request->material_or_fabric_composition,
                    'pattern'    => $request->pattern,
                    'color_name' => $request->color_name,
                    'color_code' => $request->color_code,

                    'gsm'       => $request->gsm,
                    'weight'    => $request->weight,
                    'rub_count' => $request->rub_count,
                    'moq'       => $request->moq,

                    'mrp'         => $request->mrp,
                    'offer_price' => $request->offer_price,
                    'design_repeat' => $request->design_repeat,

                    'mattress_size_category'   => $mattressSize,
                    'mattress_length_inch'     => $mattressLength,
                    'mattress_width_inch'      => $mattressWidth,
                    'mattress_thickness_inch'  => $mattressThick,
                ]
            );

            DB::commit();

            event(new SystemEvent([
                'event_type'   => 'product_price updated',
                'title'        => 'Product price updated',
                'subject'      => $variant,
                'message'      => "MRP={$variant->mrp}, Offer={$variant->offer_price}",
                'recipient_id' => User::first()?->id,
                'channels'     => ['dashboard'],
                'changes'      => $variant->toArray(),
            ]));

            return response()->json([
                'success' => true,
                'message' => 'Variant updated successfully',
                'product_variant_id' => $variant->id
            ]);
        } catch (\Throwable $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Variant update failed',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function storeImages(Request $request)
    {
        $request->validate([
            'product_variant_id' => 'required|exists:product_variants,id',
            'images.*' => 'image|max:2048',
            'primary_image' => 'required|integer'
        ]);

        // Reset old main image
        ProductImages::where('product_variant_id', $request->product_variant_id)
            ->update(['is_primary' => false]);

        if ($request->hasFile('images')) {

            foreach ($request->file('images') as $index => $file) {

                $name = time() . '-' . rand(100, 999) . '.' . $file->extension();
                $file->move(public_path('assets/upload/products'), $name);

                ProductImages::create([
                    'product_variant_id' => $request->product_variant_id,
                    'image_path' => 'assets/upload/products/' . $name,
                    'type' => 'front',
                    'is_primary' => ($index == $request->primary_image)
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Images uploaded successfully'
        ]);
    }
    public function stepThreeUpdate(Request $request)
    {
        $request->validate([
            'product_variant_id' => 'required|exists:product_variants,id',
            'images.*' => 'required|image|mimes:jpg,jpeg,png,webp|max:2048',
            'primary_image' => 'required|integer'
        ]);

        $variant = ProductVariant::with('product.category')
            ->findOrFail($request->product_variant_id);

        $slug = strtolower($variant->product->category->slug ?? '');

        /*
    |--------------------------------------------------------------------------
    | Get Target Variants
    |--------------------------------------------------------------------------
    */
        if ($slug === 'mattress') {

            $variants = ProductVariant::where(
                'product_id',
                $variant->product_id
            )->pluck('id');

            // reset all primary images for mattress product
            ProductImages::whereIn('product_variant_id', $variants)
                ->update(['is_primary' => 0]);
        } else {

            $variants = collect([$variant->id]);

            ProductImages::where(
                'product_variant_id',
                $variant->id
            )->update(['is_primary' => 0]);
        }

        /*
    |--------------------------------------------------------------------------
    | Upload Images ONCE
    |--------------------------------------------------------------------------
    */
        foreach ($request->file('images') as $index => $file) {

            $name = time() . '-' . rand(100, 999) . '.' . $file->extension();

            // 🔥 physical file stored only once
            $file->move(public_path('assets/upload/products'), $name);

            $path = 'assets/upload/products/' . $name;

            /*
        |--------------------------------------------------------------------------
        | Insert same image for all variants
        |--------------------------------------------------------------------------
        */
            foreach ($variants as $variantId) {

                ProductImages::create([
                    'product_variant_id' => $variantId,
                    'image_path' => $path,
                    'type' => 'front',
                    'is_primary' => ($index == $request->primary_image)
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => $slug === 'mattress'
                ? 'Images uploaded for all mattress variants'
                : 'Images uploaded successfully'
        ]);
    }


    // public function stepThreeUpdate(Request $request)
    // {
    //     $request->validate([
    //         'product_variant_id' => 'required|exists:product_variants,id',
    //         'images.*' => 'image|mimes:jpg,jpeg,png|max:2048',
    //     ]);

    //     $variant = ProductVariant::findOrFail($request->product_variant_id);

    //     if ($request->hasFile('images')) {

    //         foreach ($request->file('images') as $image) {

    //             $name = uniqid() . '.' . $image->extension();
    //             $image->move(public_path('uploads/products'), $name);

    //             ProductImages::create([
    //                 'product_variant_id' => $variant->id,
    //                 'image_path' => 'uploads/products/' . $name,
    //             ]);
    //         }
    //     }

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Images updated'
    //     ]);
    // }
    public function deleteVariant(Request $request)
    {
        $variant = ProductVariant::with('images')->findOrFail($request->variant_id);

        // delete images
        foreach ($variant->images as $img) {
            if (file_exists(public_path($img->image_path))) {
                unlink(public_path($img->image_path));
            }
            $img->delete();
        }

        $variant->delete();

        return response()->json([
            'success' => true,
            'message' => 'Variant deleted'
        ]);
    }

    public function deleteImage(Request $request)
    {
        $image = ProductImages::findOrFail($request->image_id);

        if (file_exists(public_path($image->image_path))) {
            unlink(public_path($image->image_path));
        }

        $image->delete();

        return response()->json(['success' => true]);
    }
    // ProductController.php

    public function updateStock(Request $request)
    {
        $validated = $request->validate([
            'stocks' => 'required|array|min:1',

            'stocks.*.id' => 'nullable|exists:stocks,id',
            'stocks.*.product_variant_id' => 'required|exists:product_variants,id',
            'stocks.*.location_id' => 'required|exists:locations,id',

            'stocks.*.opening_qty' => 'nullable|numeric|min:0',
            'stocks.*.received_qty' => 'nullable|numeric|min:0',
            'stocks.*.damaged_qty' => 'nullable|numeric|min:0',
            'stocks.*.returned_qty' => 'nullable|numeric|min:0',

            'stocks.*.available_qty' => 'required|numeric|min:0',
            'stocks.*.reorder_level' => 'nullable|numeric|min:0',
        ]);

        $savedIds = [];

        foreach ($validated['stocks'] as $stock) {

            $model = Stock::updateOrCreate(
                [
                    'id' => $stock['id'] ?? null,   // 🔥 THIS handles update safely
                ],
                [
                    'product_variant_id' => $stock['product_variant_id'],
                    'location_id'        => $stock['location_id'],

                    'opening_qty'   => $stock['opening_qty'] ?? 0,
                    'received_qty'  => $stock['received_qty'] ?? 0,
                    'damaged_qty'   => $stock['damaged_qty'] ?? 0,
                    'returned_qty'  => $stock['returned_qty'] ?? 0,

                    'available_qty' => $stock['available_qty'],
                    'reorder_level' => $stock['reorder_level'] ?? null,
                    'is_available'  => $stock['available_qty'] > 0,
                ]
            );

            $savedIds[] = $model->id;
        }

        /**
         * 🔥 DELETE REMOVED STOCKS
         * Only delete stocks NOT submitted in the form
         */
        //   Stock::whereNotIn('id', $savedIds)->delete();

        return response()->json([
            'success' => true,
            'message' => 'Stock updated successfully'
        ]);
    }


    // StockController.php
    public function destroyStock(Stock $stock)
    {
        $stock->delete();
        return response()->json(['message' => 'Stock deleted']);
    }
    public function getProductsByCategory($category_id)
    {
        $products = Product::where('category_id', $category_id)
            ->select('id', 'name')
            ->get();

        return response()->json($products);
    }
}
