<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\OrderReturn;
use Yajra\DataTables\Facades\DataTables;
use App\Notifications\SystemNotification;
use App\Services\FirebaseService;
use App\Models\Order;

class OrderReturnController extends Controller
{
    public function index()
    {
        return view('admin.orders.order_returns');
    }

    public function getOrderReturns(Request $request)
    {
        $returns = OrderReturn::with('order')->latest();

        return DataTables::of($returns)
            ->addIndexColumn()

            ->addColumn('order_number', function ($r) {
                return '
                <a href="' . route('orders.show', $r->id) . '" class="link-primary fw-medium">' . $r->order->order_number . '</a>
            ';
            })

        ->addColumn('image', function ($r) {

    if (!$r->image) {
        return '-';
    }

    // If image is string, decode manually
    $images = is_array($r->image)
        ? $r->image
        : json_decode($r->image, true);

    if (!is_array($images)) {
        return '-';
    }

    $html = '';

    foreach ($images as $img) {

        $html .= '
            <img src="'.$img.'" 
                 width="50" height="50"
                 style="object-fit:cover;border-radius:5px;cursor:pointer;margin-right:5px;"
                 onclick="previewImage(\''.$img.'\')">
        ';
    }

    return $html;
})



            ->addColumn('status', function ($r) {

                $badge = match ($r->status) {
                    'pending' => 'warning',
                    'approved' => 'success',
                    'rejected' => 'danger',
                };

                return '<span class="badge bg-' . $badge . '">'
                    . ucfirst($r->status) .
                    '</span>';
            })

            ->addColumn('action', function ($r) {
                return '
                <button class="btn btn-sm btn-primary changeStatus"
                    data-id="' . $r->id . '"
                    data-status="' . $r->status . '">
                    Change Status
                </button>

                <button class="btn btn-sm btn-danger deleteReturn"
                    data-id="' . $r->id . '">
                    Delete
                </button>';
            })

            ->rawColumns(['order_number', 'image', 'status', 'action'])
            ->make(true);
    }


public function updateStatus(Request $request)
{
    $request->validate([
        'id' => 'required|exists:order_returns,id',
        'status' => 'required|in:pending,approved,rejected'
    ]);

    $return = OrderReturn::with('order.user')->findOrFail($request->id);

    $return->status = $request->status;
    $return->save();

    $order = $return->order;
    $user  = $order->user;

    $title = '';
    $message = '';
    $eventType = '';
    $subjectId = $order->id;
    $subjectType = Order::class;

    /*
    |--------------------------------------------------------------------------
    | If Approved → Update Order Status to Returned
    |--------------------------------------------------------------------------
    */

    if ($request->status === 'approved') {

        $order->update([
            'status' => 'returned'
        ]);

        $title = 'Return Approved';
        $message = "✅ Your return request for Order #{$order->order_number} has been approved. Refund will be processed soon.";
        $eventType = 'return_approved';
    }

    /*
    |--------------------------------------------------------------------------
    | If Rejected
    |--------------------------------------------------------------------------
    */

    if ($request->status === 'rejected') {

        $title = 'Return Rejected';
        $message = "❌ Your return request for Order #{$order->order_number} has been rejected.";
        $eventType = 'return_rejected';
    }

    /*
    |--------------------------------------------------------------------------
    | Send Database Notification
    |--------------------------------------------------------------------------
    */

    if ($user && $request->status !== 'pending') {

        $user->notify(new SystemNotification(
            $title,
            $message,
            $eventType,
            $subjectId,
            $subjectType
        ));

        /*
        |--------------------------------------------------------------------------
        | Send Firebase Push
        |--------------------------------------------------------------------------
        */

        if (!empty($user->fcm_token)) {

            app(FirebaseService::class)->sendPush(
                $user->fcm_token,
                $title,
                $message,
                [
                    'event_type'   => $eventType,
                    'subject_id'   => (string)$subjectId,
                    'subject_type' => $subjectType,
                ]
            );
        }
    }

    return response()->json([
        'success' => true,
        'message' => 'Return status updated successfully'
    ]);
}


    public function destroy($id)
    {
        OrderReturn::findOrFail($id)->delete();

        return response()->json(['success' => 'Return request deleted successfully']);
    }
}
