<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use Yajra\DataTables\Facades\DataTables;
use Carbon\Carbon;
use App\Models\OrderItem;
use App\Models\OrderStatusHistory;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\DB;
use App\Notifications\SystemNotification;
use App\Models\WholesaleEnquiry;


class OrderController extends Controller
{
    public function index()
    {
        return view('admin.orders.list');
    }



    public function list(Request $request)
    {
        $query = Order::with(['items.product', 'courier', 'shippingAddress'])
            ->select('orders.*')->orderBy('id', 'DESC');

        // DATE FILTER
        if ($request->from_date && $request->to_date) {

    $from = Carbon::parse($request->from_date)->startOfDay();   // 00:00:00
    $to   = Carbon::parse($request->to_date)->endOfDay();       // 23:59:59

    $query->whereBetween('created_at', [$from, $to]);
}
        // STATUS FILTER
        if ($request->status) {
            $query->where('status', $request->status);
        }

        // CATEGORY FILTER
        if ($request->category_id) {
            $query->whereHas('items.product', function ($q) use ($request) {
                $q->where('category_id', $request->category_id);
            });
        }

        // SUBCATEGORY FILTER
        if ($request->subcategory_id) {
            $query->whereHas('items.product', function ($q) use ($request) {
                $q->where('subcategory_id', $request->subcategory_id);
            });
        }
     if ($request->order_type == 'project') {
        $query->whereNotNull('quotation_id');
    }
        return DataTables::of($query)

            ->addIndexColumn()

            ->addColumn('customer', function ($row) {
                return optional($row->shippingAddress)->name ?? '-';
            })
            ->addColumn('order_number', function ($row) {
                return '
                <a href="' . route('orders.show', $row->id) . '" class="text-dark btn-sm">' . $row->order_number . '</a>
            ';
            })
            ->addColumn('items_count', function ($row) {
                return $row->items->count();
            })

            ->addColumn('delivery_number', function ($row) {
                return $row->tracking_id ?? '-';
            })

            ->editColumn('total_amount', function ($row) {
                return '₹ ' . number_format($row->total_amount, 2);
            })

            ->editColumn('payment_status', function ($row) {
                if ($row->payment_status == 'paid') {
                    return '<span class="badge bg-success">Paid</span>';
                }
                return '<span class="badge bg-warning">Pending</span>';
            })

            ->editColumn('status', function ($row) {
                return '<span class="badge bg-info">' . $row->status . '</span>';
            })

            ->addColumn('action', function ($row) {
                return '
                <a href="' . route('orders.show', $row->id) . '" class="btn btn-success btn-sm">View</a>
            ';
            })

            ->rawColumns(['order_number', 'payment_status', 'status', 'action'])

            ->make(true);
    }
public function show($id)
{
    $order = Order::with([
        'project',
        'items.product',
        'items.variant.primaryImage',
        'statusHistory',
        'shippingAddress',
        'billingAddress',
        'courier',
        'user'
    ])->findOrFail($id);

 return view('admin.orders.show', compact('order'));

//     return view('admin.orders.show', compact('order'));
 }


    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|string',
            'payment_status' => 'required|in:pending,success,failed'
        ]);

        DB::beginTransaction();
        try {
            $order = Order::findOrFail($id);

            // Update status + payment_status
            $order->update([
                'status' => $request->status,
                'payment_status' => $request->payment_status,
            ]);

            // Insert Status History
            OrderStatusHistory::create([
                'order_id' => $order->id,
                'status' => $request->status,
                'note' => $request->note ?? null,
                'created_at' => now(),
            ]);

            DB::commit();
            $user = $order->user;
            switch ($request->status) {

                case 'placed':
                    $title = 'Order Placed';
                    $message = "Your order #{$order->order_number} has been placed successfully.";
                    $eventType = 'order_placed';
                    break;

                case 'processing':
                    $title = 'Order Processing';
                    $message = "Good news! Your order #{$order->order_number} is now being prepared.";
                    $eventType = 'order_processing';
                    break;

                case 'shipped':
                    $title = 'Order Shipped';
                    $message = "🚚 Your order #{$order->order_number} has been shipped and is on the way.";
                    $eventType = 'order_shipped';
                    break;

                case 'delivered':
                    $title = 'Order Delivered';
                    $message = "✅ Your order #{$order->order_number} has been delivered successfully.";
                    $eventType = 'order_delivered';
                    break;

                case 'cancelled':
                    $title = 'Order Cancelled';
                    $message = "❌ Your order #{$order->order_number} has been cancelled.";
                    $eventType = 'order_cancelled';
                    break;

                default:
                    $title = 'Order Updated';
                    $message = "Your order #{$order->order_number} status updated to {$request->status}.";
                    $eventType = 'order_updated';
            }

            if ($request->payment_status == 'success') {
                $message .= " Payment received successfully 💰.";
            }

            if ($request->payment_status == 'failed') {
                $message .= " ⚠️ Payment failed. Please retry.";
            }


            $subjectId = $order->id;
            $subjectType = 'App\\Models\\Order';

            $user->notify(new SystemNotification(
                $title,
                $message,
                $eventType,
                $subjectId,
                $subjectType
            ));

            return response()->json([
                'success' => true,
                'message' => 'Order and payment status updated successfully'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function downloadInvoice($id)
    {
        $order = Order::with([
            'items.variant',
            'billingAddress',
            'shippingAddress',
            //   'user'
        ])->findOrFail($id);

        //return view('pdf.order_invoice',compact('order'));

        $pdf = PDF::loadView('pdf.order_invoice', compact('order'));

        return $pdf->download('invoice_' . $order->order_number . '.pdf');
    }

    public function updateTracking(Request $request)
    {
        $order = Order::findOrFail($request->order_id);

        $order->update([
            'tracking_id' => $request->tracking_id,
            'expected_delivery_date' => $request->expected_delivery_date,
            'delivered_at' => $request->delivered_at,
        ]);

        // ✅ Get Order User
        $user = $order->user;

        // ✅ Dynamic Notification Message
        if ($request->delivered_at) {

            $title = 'Order Delivered';
            $message = "✅ Your order #{$order->order_number} has been delivered successfully.";
            $eventType = 'order_delivered';
        } elseif ($request->tracking_id) {

            $title = 'Tracking Updated';
            $message = "🚚 Tracking ID {$request->tracking_id} added for your order #{$order->order_number}.";

            if ($request->expected_delivery_date) {
                $message .= " Expected delivery: " . date('d M Y', strtotime($request->expected_delivery_date)) . ".";
            }

            $eventType = 'tracking_updated';
        } else {

            $title = 'Order Update';
            $message = "Your order #{$order->order_number} tracking information updated.";
            $eventType = 'order_update';
        }

        // ✅ Send Notification
        if ($user) {
            $user->notify(new SystemNotification(
                $title,
                $message,
                $eventType,
                $order->id,
                'App\\Models\\Order'
            ));
        }

        return response()->json([
            'success' => true,
            'message' => 'Tracking Updated Successfully'
        ]);
    }


    public function deleteOrder(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id'
        ]);

        $order = Order::findOrFail($request->order_id);
        $order->delete();

        return response()->json([
            'success' => true,
            'message' => 'Order deleted successfully'
        ]);
    }

    public function getWholesaleEnquiry()
    {
        $data = WholesaleEnquiry::latest();

        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn('action', function ($row) {

                return '<button class="btn btn-danger btn-sm deleteEnquiry"
                    data-id="' . $row->id . '">Delete</button>';
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function deleteWholesaleEnquiry($id)
{
    WholesaleEnquiry::findOrFail($id)->delete();

    return response()->json([
        'success' => 'Enquiry deleted successfully'
    ]);
}

}
