<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Coupon;
use App\Models\Category;
use App\Models\Product;
use Yajra\DataTables\Facades\DataTables;

class CouponController extends Controller
{
    // List view
    public function index()
    {
        return view('admin.coupons');
    }

    // Store Coupon
    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string|max:50|unique:coupons,code',
            'type' => 'required|in:percentage,fixed',
            'value' => 'required|numeric|min:0',

            'max_discount_amount' => 'nullable|numeric|min:0',

            'category_id' => 'nullable|exists:categories,id',
            'product_id' => 'nullable|exists:products,id',

            'min_purchase' => 'nullable|numeric|min:0',
            'max_uses' => 'nullable|integer|min:1',
            'per_user_limit' => 'nullable|integer|min:1',

            'allowed_roles' => 'nullable|array',
            'allowed_roles.*' => 'string',

            'first_time_user_only' => 'nullable|boolean',
            'is_stackable' => 'nullable|boolean',
            'auto_apply' => 'nullable|boolean',
            'is_referral' => 'nullable|boolean',

            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',

            'description' => 'nullable|string|max:255',
        ]);

        $validated['allowed_roles'] = $request->allowed_roles
            ? json_encode($request->allowed_roles)
            : null;

        Coupon::create($validated);

        return response()->json(['success' => 'Coupon added successfully']);
    }

    // Datatable list
    public function getCoupons()
    {
        $coupons = Coupon::with(['category', 'product'])->orderByDesc('id');

        return DataTables::of($coupons)
            ->addIndexColumn()
            ->addColumn('code', fn($c) => strtoupper($c->code))
            ->addColumn('type_value', function ($c) {
                return $c->type === 'percentage'
                    ? $c->value . '% (Max ₹' . ($c->max_discount_amount ?? '-') . ')'
                    : '₹' . $c->value;
            })
            ->addColumn('category', fn($c) => $c->category?->name ?? '-')
            ->addColumn('product', fn($c) => $c->product?->name ?? '-')
            ->addColumn('usage', fn($c) => $c->used_count . '/' . ($c->max_uses ?? '∞'))
            ->addColumn('date_range', fn($c) => ($c->start_date ?? '-') . ' to ' . ($c->end_date ?? '-'))
            ->addColumn('status', function ($c) {
                $btnClass = $c->is_active ? 'btn-success' : 'btn-danger';
                $btnText = $c->is_active ? 'Published' : 'Unpublished';
                return '<button class="btn btn-sm toggleStatusBtn ' . $btnClass . '" data-id="' . $c->id . '">' . $btnText . '</button>';
            })
            ->addColumn('action', function ($row) {
                return '
                    <button class="btn btn-sm btn-purple editCouponBtn" data-id="' . $row->id . '">Edit</button>
                    <button class="btn btn-sm btn-danger deleteCoupon" data-id="' . $row->id . '">Delete</button>
                ';
            })
            ->escapeColumns([])
            ->make(true);
    }

    // Edit coupon
    public function edit($id)
    {
       
        $coupon = Coupon::findOrFail($id);

        $coupon->allowed_roles = $coupon->allowed_roles
            ? json_decode($coupon->allowed_roles)
            : [];

        return response()->json(['data' => $coupon]);
    }

    // Update Coupon
    public function update(Request $request, $id)
    {
        $coupon = Coupon::findOrFail($id);

        $validated = $request->validate([
            'code' => 'required|string|max:50|unique:coupons,code,' . $coupon->id,
            'type' => 'required|in:percentage,fixed',
            'value' => 'required|numeric|min:0',

            'max_discount_amount' => 'nullable|numeric|min:0',

            'category_id' => 'nullable|exists:categories,id',
            'product_id' => 'nullable|exists:products,id',

            'min_purchase' => 'nullable|numeric|min:0',
            'max_uses' => 'nullable|integer|min:1',
            'per_user_limit' => 'nullable|integer|min:1',

            'allowed_roles' => 'nullable|array',
            'allowed_roles.*' => 'string',

            'first_time_user_only' => 'nullable|boolean',
            'is_stackable' => 'nullable|boolean',
            'auto_apply' => 'nullable|boolean',
            'is_referral' => 'nullable|boolean',

            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',

            'is_active' => 'boolean',
            'description' => 'nullable|string|max:255',
        ]);

        $validated['allowed_roles'] = $request->allowed_roles
            ? json_encode($request->allowed_roles)
            : null;

        $coupon->update($validated);

        return response()->json(['success' => true, 'message' => 'Coupon updated successfully']);
    }

    // Delete coupon
    public function destroy($id)
    {
        Coupon::findOrFail($id)->delete();
        return response()->json(['success' => 'Coupon deleted successfully']);
    }

    // Toggle status
    public function changeStatus($id)
    {
        $coupon = Coupon::findOrFail($id);
        $coupon->is_active = !$coupon->is_active;
        $coupon->save();

        return response()->json([
            'success' => true,
            'status' => $coupon->is_active
        ]);
    }

    // Get products by category
    public function getProducts($category_id)
    {
        return Product::where('category_id', $category_id)
            ->where('is_active', 1)
            ->select('id', 'name')
            ->get();
    }
}
