<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use App\Models\Banner;
use Yajra\DataTables\Facades\DataTables;
use Carbon\Carbon;
use App\Events\SystemEvent;
use App\Models\User;

class BannerController extends Controller
{
    // =========================
    // ADD BANNER
    // =========================
    public function addBanner(Request $request)
    {
        $request->validate([
            'title'       => 'nullable|string|max:255',

            'banner_type' => 'required|in:home_main,offer,new_launch,festival,announcement',
            'placement'   => 'required|in:home,category,product,dashboard',
            'audience'    => 'required|in:all,b2c,b2b,dealer,franchise,architect',
            'platform'    => 'required|in:all,website,b2c_app,b2b_app',

            'priority'    => 'nullable|integer|min:0',
            'auto_sort'   => 'nullable|boolean',

            'category_id' => 'nullable|exists:categories,id',
            'product_id'  => 'nullable|exists:products,id',

            'image'       => 'required|image|mimes:jpg,jpeg,png,webp|max:4096',

            // 🔥 Only for Festival banner
            'start_date' => 'nullable|date',
            'end_date'   => 'nullable|date',
        ]);

        // Upload Image
        $imagePath = null;
        if ($request->hasFile('image')) {
            $image = $request->image;
            $imageName = time() . '-' . rand(100, 999) . '.' . $image->extension();
            $uploadPath = public_path('assets/upload/banner/');

            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0755, true);
            }

            $image->move($uploadPath, $imageName);
            $imagePath = 'assets/upload/banner/' . $imageName;
        }

        Banner::create([
            'title'       => ucfirst($request->title),
            'banner_type' => $request->banner_type,
            'placement'   => $request->placement,
            'audience'    => $request->audience,
            'platform'    => $request->platform,

            'priority'    => $request->priority ?? 0,
            'auto_sort'   => $request->auto_sort ?? 0,

            'category_id' => $request->category_id,
            'product_id'  => $request->product_id,

            'image'       => $imagePath,

            'start_date' => $request->start_date
                ? Carbon::parse($request->start_date)->format('Y-m-d')
                : null,
            'end_date' => $request->end_date
                ? Carbon::parse($request->end_date)->format('Y-m-d')
                : null,
            // 'start_date' => $request->banner_type === 'festival' && $request->start_date
            //     ? Carbon::parse($request->start_date)->format('Y-m-d')
            //     : null,

            // 'end_date' => $request->banner_type === 'festival' && $request->end_date
            //     ? Carbon::parse($request->end_date)->format('Y-m-d')
            //     : null,

            'views'       => 0,
            'clicks'      => 0,
            'status'      => 1,
        ]);

        return response()->json(['success' => 'Banner added successfully!']);
    }

    // =========================
    // GET BANNERS (DATATABLE)
    // =========================
  public function getBanner()
{
    $banners = Banner::orderBy('priority', 'ASC')
        ->orderBy('clicks', 'DESC')
        ->latest()
        ->get();

    return DataTables::of($banners)
        ->addIndexColumn()

        // 🔹 TITLE + IMAGE COLUMN
        ->addColumn('title_image', function ($b) {
            $img = $b->image_url
                ? '<img src="'.$b->image_url.'" 
                        style="width:70px;height:40px;object-fit:cover;border-radius:6px;margin-right:8px;">'
                : '';

            return '
                <div style="display:flex;align-items:center;">
                    '.$img.'
                    <span>'.$b->title.'</span>
                </div>
            ';
        })

        ->addColumn('status', fn($b) =>
            '<button class="btn btn-sm toggleStatusBtn ' .
                ($b->status ? 'btn-success' : 'btn-danger') .
                '" data-id="' . $b->id . '">' .
                ($b->status ? 'Published' : 'Unpublished') .
            '</button>'
        )

        ->addColumn('action', function ($b) {
            return '
            <button class="btn btn-sm btn-purple editbanner"
                data-id="'.$b->id.'"
                data-title="'.e($b->title).'"
                data-banner_type="'.$b->banner_type.'"
                data-placement="'.$b->placement.'"
                data-audience="'.$b->audience.'"
                data-platform="'.$b->platform.'"
                data-category_id="'.$b->category_id.'"
                data-product_id="'.$b->product_id.'"
                data-priority="'.$b->priority.'"
                data-auto_sort="'.$b->auto_sort.'"
                data-start_date="'.$b->start_date.'"
                data-end_date="'.$b->end_date.'"
                data-status="'.$b->status.'"
                data-image="'.$b->image_url.'"
            >Edit</button>

            <button class="btn btn-sm btn-danger deletebanner" data-id="'.$b->id.'">
                Delete
            </button>';
        })

        ->rawColumns(['title_image', 'status', 'action'])
        ->make(true);
}

    // =========================
    // UPDATE BANNER
    // =========================
    public function update(Request $request, $id)
    {
        $banner = Banner::findOrFail($id);

        $request->validate([
            'title'       => 'nullable|string|max:255',

            'banner_type' => 'required|in:home_main,offer,new_launch,festival,announcement',
            'placement'   => 'required|in:home,category,product,dashboard',
            'audience'    => 'required|in:all,b2c,b2b,dealer,franchise,architect',
            'platform'    => 'required|in:all,website,b2c_app,b2b_app',

            'priority'    => 'nullable|integer|min:0',
            'auto_sort'   => 'nullable|boolean',

            'category_id' => 'nullable|exists:categories,id',
            'product_id'  => 'nullable|exists:products,id',

            'image'       => 'nullable|image|mimes:jpg,jpeg,png,webp|max:4096',

            'start_date' => 'nullable|date',
            'end_date'   => 'nullable|date',
        ]);

        // Image Update
        $imagePath = $banner->image;
        if ($request->hasFile('image')) {
            if ($banner->image && File::exists(public_path($banner->image))) {
                File::delete(public_path($banner->image));
            }

            $image = $request->image;
            $imageName = time() . '-' . rand(100, 999) . '.' . $image->extension();
            $uploadPath = public_path('assets/upload/banner/');
            $image->move($uploadPath, $imageName);
            $imagePath = 'assets/upload/banner/' . $imageName;
        }

        $banner->update([
            'title'       => ucfirst($request->title),
            'banner_type' => $request->banner_type,
            'placement'   => $request->placement,
            'audience'    => $request->audience,
            'platform'    => $request->platform,

            'priority'    => $request->priority ?? 0,
            'auto_sort'   => $request->auto_sort ?? 0,

            'category_id' => $request->category_id,
            'product_id'  => $request->product_id,

            'image'       => $imagePath,

            //  'start_date'  => $request->banner_type === 'festival' && $request->start_date
            //                             ? Carbon::parse($request->start_date)->format('Y-m-d')
            //                             : null,
            //         'end_date'    => $request->banner_type === 'festival' && $request->end_date
            //                             ? Carbon::parse($request->end_date)->format('Y-m-d')
            //                             : null,
            'start_date' => $request->start_date
                ? Carbon::parse($request->start_date)->format('Y-m-d')
                : null,
            'end_date' => $request->end_date
                ? Carbon::parse($request->end_date)->format('Y-m-d')
                : null,
        ]);

        return response()->json(['success' => 'Banner updated successfully!']);
    }

    // =========================
    // DELETE BANNER
    // =========================
    public function destroy($id)
    {
        $banner = Banner::findOrFail($id);

        if ($banner->image && File::exists(public_path($banner->image))) {
            File::delete(public_path($banner->image));
        }

        $banner->delete();

        return response()->json(['success' => 'Banner deleted successfully.']);
    }

    // =========================
    // TOGGLE STATUS
    // =========================
    public function changeStatus($id)
    {
        $banner = Banner::findOrFail($id);
        $banner->status = !$banner->status;
        $banner->save();

        //*** notification ********/
        $adminId = User::first()?->id;

        event(new SystemEvent([
            'event_type'   => 'product_created',
            'subject'      =>  $banner,
            'message'      => "Banner  status changed to " . $banner->status,
            'recipient_id' => User::first()?->id,
            'channels'     => ['dashboard'],
            'changes'      =>  $banner->toArray(),
        ]));

        //*** notification end********/
        return response()->json(['success' => true, 'status' => $banner->status]);
    }
}
