<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Room;
use Carbon\Carbon;
use App\Models\ProjectArea;
use App\Models\Project;
use App\Models\ProjectItem;
use App\Models\Category;
use App\Models\SubCategory;
use App\Models\ProjectItemAccessories;
use App\Models\ProjectItemCurtain;
use App\Models\ProjectItemMattress;
use App\Models\ProjectItemMeasurement;
use App\Models\ProjectItemSofa;
use App\Models\ProjectItemWallpaper;
use App\Models\ProductVariant;
use Illuminate\Support\Facades\DB;
use App\Notifications\SystemNotification;
use App\Models\User;
use Illuminate\Support\Facades\File;
use App\Models\Quotation;
use PDF;

class ProjectController extends Controller
{
    public function store(Request $request)
    {
        $validated = $request->validate([
            'project_name' => 'required|string|max:255',
            'customer_name' => 'required|string|max:255',
            'customer_phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',

            'location_lat' => 'nullable|numeric',
            'location_lng' => 'nullable|numeric',

            'expected_completion_date' => 'nullable|date',

            'employee_name' => 'nullable|string|max:255',
            'employee_contact' => 'nullable|string|max:20',

            'project_type' => 'nullable|in:home,villa,flat,commercial'
        ]);

        // ✅ Auto Project Number
        $projectNumber = 'CUS-' . date('Ymd') . '-' . rand(1000, 9999);

        $user = auth()->user();

        $dealerId = null;
        $architectId = null;

        switch ($user->user_type) {

            case 'dealer':
                $dealerId = $user->id;
                break;

            case 'architect':
                $architectId = $user->id;
                break;

            case 'dealer_staff':
                $dealerId = $user->dealer_id;
                break;

            case 'architect_staff':
                $architectId = $user->architect_id;
                break;
        }



        $project = Project::create([
            'project_number' => $projectNumber,
            'project_name' => $validated['project_name'],
            'customer_id' => NULL,
            'customer_name' => $validated['customer_name'],
            'customer_phone' => $validated['customer_phone'] ?? null,

            'address' => $validated['address'] ?? null,
            'location_lat' => $validated['location_lat'] ?? null,
            'location_lng' => $validated['location_lng'] ?? null,

            'expected_completion_date' => date("Y-m-d", strtotime($validated['expected_completion_date'])) ?? null,

            'employee_name' => $validated['employee_name'] ?? null,
            'employee_contact' => $validated['employee_contact'] ?? null,

            'project_type' => $validated['project_type'] ?? null,
            // ✅ Auto assign
            'dealer_id' => $dealerId,
            'architect_id' => $architectId,

            'created_by' => auth()->id(),
        ]);



        $admin = User::first();
        $adminId = $admin?->id;
        event(new \App\Events\SystemEvent([
            'type' => 'project',
            'title'        => 'New Project',
            'subject' => $project,
            'message' => "New project {$project->project_number} created by {$project->employee_name}",
            'recipient_id' => $adminId, // Admin to notify
            'channels' => ['dashboard'],
        ]));

        return response()->json([
            'success' => true,
            'message' => 'Project created successfully',
            'data' => $project
        ]);
    }
    public function editProject(Request $request)
    {
        // Validate request
        $validated = $request->validate([
            'project_id' => 'required|exists:projects,id',
            'project_name' => 'required|string|max:255',
            'customer_name' => 'required|string|max:255',
            'customer_phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',

            'location_lat' => 'nullable|numeric',
            'location_lng' => 'nullable|numeric',

            'expected_completion_date' => 'nullable|date',

            'employee_name' => 'nullable|string|max:255',
            'employee_contact' => 'nullable|string|max:20',

            'project_type' => 'nullable|in:home,villa,flat,commercial'
        ]);

        // Find project
        $project = Project::findOrFail($validated['project_id']);

        // Update project
        $project->update([
            'project_name' => $validated['project_name'],
            'customer_name' => $validated['customer_name'],
            'customer_phone' => $validated['customer_phone'] ?? null,
            'address' => $validated['address'] ?? null,
            'location_lat' => $validated['location_lat'] ?? null,
            'location_lng' => $validated['location_lng'] ?? null,
            'expected_completion_date' => isset($validated['expected_completion_date']) ? date("Y-m-d", strtotime($validated['expected_completion_date'])) : null,
            'employee_name' => $validated['employee_name'] ?? null,
            'employee_contact' => $validated['employee_contact'] ?? null,
            'project_type' => $validated['project_type'] ?? null,
            'updated_by' => auth()->id(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Project updated successfully',
            'data' => $project
        ]);
    }

    // public function storeProjectItem(Request $request)
    // {
    //     // print_r($request->details);die();
    //     $validated = $request->validate([
    //         'category' => 'required|in:curtain,sofa,mattress,wallpaper,accessories',
    //         'product_id' => 'nullable|exists:products,id',
    //         'variant_id' => 'nullable|exists:product_variants,id',
    //         'quantity' => 'nullable|integer|min:1',

    //         'measurement.width' => 'nullable|numeric',
    //         'measurement.height' => 'nullable|numeric',
    //         'measurement.length' => 'nullable|numeric',
    //         'measurement.size' => 'nullable|string',

    //         'details' => 'nullable|array'
    //     ]);
    //     $projectId = $request->project_id;
    //     $areaId = $request->area_id;
    //     DB::beginTransaction();

    //     $item = ProjectItem::create([
    //         'project_id' => $projectId,
    //         'area_id' => $areaId,
    //         'category' => $validated['category'],
    //         'product_id' => $validated['product_id'] ?? null,
    //         'variant_id' => $validated['variant_id'] ?? null,
    //         'quantity' => $validated['quantity'] ?? 1
    //     ]);

    //     // Measurement
    //     if ($request->measurement) {
    //         ProjectItemMeasurement::create([
    //             'project_item_id' => $item->id,
    //             ...$request->measurement
    //         ]);
    //     }

    //     // CATEGORY SWITCH
    //     switch ($validated['category']) {

    //         case 'curtain':
    //             ProjectItemCurtain::create([
    //                 'project_item_id' => $item->id,
    //                 ...$request->details
    //             ]);
    //             break;

    //         case 'sofa':
    //             ProjectItemSofa::create([
    //                 'project_item_id' => $item->id,
    //                 ...$request->details
    //             ]);
    //             break;

    //         case 'mattress':
    //             ProjectItemMattress::create([
    //                 'project_item_id' => $item->id,
    //                 ...$request->details
    //             ]);
    //             break;

    //         case 'wallpaper':
    //             ProjectItemWallpaper::create([
    //                 'project_item_id' => $item->id,
    //                 ...$request->details
    //             ]);
    //             break;

    //         case 'accessories':
    //             ProjectItemAccessories::create([
    //                 'project_item_id' => $item->id,
    //                 ...$request->details
    //             ]);
    //             break;
    //     }

    //     DB::commit();

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Item added successfully'
    //     ]);
    // }


    // public function storeProjectItem(Request $request)
    // {

    //     $validated = $request->validate([
    //         'project_item_id' => 'nullable|exists:project_items,id',
    //         'category' => 'required|in:curtain,sofa,mattress,wallpaper,accessories',
    //         'product_id' => 'nullable|exists:products,id',
    //         'variant_id' => 'nullable|exists:product_variants,id',
    //         'quantity' => 'nullable|integer|min:1',
    //         'note' => 'nullable|string',
    //         'measurement' => 'nullable|array',
    //         'measurement.width' => 'nullable|numeric',
    //         'measurement.height' => 'nullable|numeric',
    //         'measurement.length' => 'nullable|numeric',
    //         'measurement.size' => 'nullable|string',
    //         'measurement.measurement_type' => 'nullable|string',
    //         'measurement.unit' => 'nullable|string',

    //         'details' => 'nullable|array',
    //         'items' => 'nullable|array|min:1',
    //         'items.*.product_id' => 'nullable|exists:products,id',
    //         'items.*.variant_id' => 'nullable|exists:product_variants,id',
    //         'items.*.quantity' => 'nullable|integer|min:1',
    //         'items.*.note' => 'nullable|string', // ✅ ADD THIS
    //     ]);

    //     //  print_r($request->all());die();
    //     $projectId = $request->project_id;
    //     $areaId = $request->area_id;
    //     DB::beginTransaction();
    //     if ($validated['category'] != 'sofa') {

    //         $item = ProjectItem::create([
    //             'project_id' => $projectId,
    //             'area_id' => $areaId,
    //             'category' => $validated['category'],
    //             'product_id' => $validated['product_id'] ?? null,
    //             'variant_id' => $validated['variant_id'] ?? null,
    //             'quantity' => $validated['quantity'] ?? 1,
    //             'notes' => $validated['note'] ?? null

    //         ]);
    //     }
    //     // Measurement
    //     if (!empty($validated['items'])) {

    //         foreach ($validated['items'] as $row) {

    //             $item = ProjectItem::create([
    //                 'project_id' => $projectId,
    //                 'area_id'    => $areaId,
    //                 'category'   => $validated['category'],
    //                 'product_id' => $row['product_id'] ?? null,
    //                 'variant_id' => $row['variant_id'] ?? null,
    //                 'quantity'   => $row['quantity'] ?? 1,
    //                 'notes'       => $row['note'] ?? null, // ✅ FIX
    //             ]);

    //             // ✅ SOFA AUTO MEASUREMENT FROM VARIANT
    //             if ($validated['category'] === 'sofa' && !empty($row['variant_id'])) {

    //                 $variant = ProductVariant::find($row['variant_id']);

    //                 if ($variant && $variant->dimensions) {

    //                     $dim = is_string($variant->dimensions)
    //                         ? json_decode($variant->dimensions, true)
    //                         : $variant->dimensions;

    //                     if (is_array($dim)) {
    //                         ProjectItemMeasurement::create([
    //                             'project_item_id' => $item->id,
    //                             'length' => $dim['length'] ?? null,
    //                             'width'  => $dim['width'] ?? null,
    //                             'height' => $dim['height'] ?? null,
    //                             'measurement_type' => 'sofa',
    //                             'unit' => $variant->unit ?? 'cm'
    //                         ]);
    //                     }
    //                 }
    //             }
    //         }
    //     }


    //     if ($request->measurement) {
    //         ProjectItemMeasurement::create([
    //             'project_item_id' => $item->id,
    //             ...$request->measurement
    //         ]);
    //     }

    //     // CATEGORY SWITCH
    //     switch ($validated['category']) {

    //         case 'curtain':
    //             ProjectItemCurtain::create([
    //                 'project_item_id' => $item->id,
    //                 ...$request->details
    //             ]);
    //             break;

    //         case 'mattress':
    //             ProjectItemMattress::create([
    //                 'project_item_id' => $item->id,
    //                 ...$request->details
    //             ]);
    //             break;

    //         case 'wallpaper':
    //             ProjectItemWallpaper::create([
    //                 'project_item_id' => $item->id,
    //                 ...$request->details
    //             ]);
    //             break;

    //             // case 'accessories':
    //             //     ProjectItemAccessories::create([
    //             //         'project_item_id' => $item->id,
    //             //         ...$request->details
    //             //     ]);
    //             //     break;
    //     }

    //     DB::commit();

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Item added successfully'
    //     ]);
    // }

    public function storeProjectItem(Request $request)
    {
        $validated = $request->validate([
            'project_item_id' => 'nullable|exists:project_items,id',
            'category' => 'required',
            'product_id' => 'nullable|exists:products,id',
            'variant_id' => 'nullable|exists:product_variants,id',
            'quantity' => 'nullable|integer|min:1',
            'note' => 'nullable|string',
            'measurement' => 'nullable|array',
            'details' => 'nullable|array',
            'items' => 'nullable|array|min:1',
            'items.*.project_item_id' => 'nullable|exists:project_items,id',
            'items.*.product_id' => 'nullable|exists:products,id',
            'items.*.variant_id' => 'nullable|exists:product_variants,id',
            'items.*.quantity' => 'nullable|integer|min:1',
            'items.*.note' => 'nullable|string',

        ]);

        DB::beginTransaction();

        try {

            $projectId = $request->project_id;
            $areaId    = $request->area_id;

            /* =========================
           SINGLE ITEM (NON SOFA)
        ========================= */

            if ($validated['category'] != 'sofa') {
                if ($validated['category'] != 'sofa') {

                    $item = ProjectItem::updateOrCreate(
                        ['id' => $validated['project_item_id'] ?? null],
                        [
                            'project_id' => $projectId,
                            'area_id'    => $areaId,
                            'category'   => $validated['category'],
                            'product_id' => $validated['product_id'] ?? null,
                            'variant_id' => $validated['variant_id'] ?? null,
                            'quantity'   => $validated['quantity'] ?? 1,
                            'notes'      => $validated['note'] ?? null,
                        ]
                    );
                }


                // if (!empty($row['project_item_id'])) {
                //     $item = ProjectItem::find($row['project_item_id']);
                //     $item->update([
                //         'product_id' => $row['product_id'] ?? null,
                //         'variant_id' => $row['variant_id'] ?? null,
                //         'quantity'   => $row['quantity'] ?? 1,
                //         'notes'      => $row['note'] ?? null,
                //     ]);
                // } else {
                //     $item = ProjectItem::create([
                //         'project_id' => $projectId,
                //         'area_id'    => $areaId,
                //         'category'   => 'sofa',
                //         'product_id' => $row['product_id'] ?? null,
                //         'variant_id' => $row['variant_id'] ?? null,
                //         'quantity'   => $row['quantity'] ?? 1,
                //         'notes'      => $row['note'] ?? null,
                //     ]);
                // }
            }

            /* =========================
           MULTIPLE ITEMS (SOFA)
        ========================= */

            if (!empty($validated['items'])) {

                foreach ($validated['items'] as $row) {

                    $item = ProjectItem::updateOrCreate(
                        ['id' => $row['project_item_id'] ?? null],
                        [
                            'project_id' => $projectId,
                            'area_id'    => $areaId,
                            'category'   => $validated['category'],
                            'product_id' => $row['product_id'] ?? null,
                            'variant_id' => $row['variant_id'] ?? null,
                            'quantity'   => $row['quantity'] ?? 1,
                            'notes'      => $row['note'] ?? null,
                        ]
                    );

                    /* AUTO MEASUREMENT FOR SOFA */
                    if ($validated['category'] === 'sofa' && !empty($row['variant_id'])) {

                        $variant = ProductVariant::find($row['variant_id']);

                        if ($variant && $variant->dimensions) {

                            $dim = is_string($variant->dimensions)
                                ? json_decode($variant->dimensions, true)
                                : $variant->dimensions;

                            ProjectItemMeasurement::updateOrCreate(
                                ['project_item_id' => $item->id],
                                [
                                    'length' => $dim['length'] ?? null,
                                    'width'  => $dim['width'] ?? null,
                                    'height' => $dim['height'] ?? null,
                                    'measurement_type' => 'sofa',
                                    'unit' => $variant->unit ?? 'cm'
                                ]
                            );
                        }
                    }
                }
            }

            /* =========================
           MANUAL MEASUREMENT
        ========================= */

            if ($request->measurement && isset($item)) {

                ProjectItemMeasurement::updateOrCreate(
                    ['project_item_id' => $item->id],
                    $request->measurement
                );
            }

            /* =========================
           CATEGORY DETAILS
        ========================= */

            if (!empty($request->details) && isset($item)) {

                switch ($validated['category']) {

                    case 'curtain':
                        ProjectItemCurtain::updateOrCreate(
                            ['project_item_id' => $item->id],
                            $request->details
                        );
                        break;

                    case 'mattress':
                        ProjectItemMattress::updateOrCreate(
                            ['project_item_id' => $item->id],
                            $request->details
                        );
                        break;

                    case 'wallpaper':
                        ProjectItemWallpaper::updateOrCreate(
                            ['project_item_id' => $item->id],
                            $request->details
                        );
                        break;
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => isset($validated['project_item_id'])
                    ? 'Item updated successfully'
                    : 'Item added successfully'
            ]);
        } catch (\Exception $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    public function storeProjectArea(Request $request)
    {
        $request->validate([
            'room_id' => 'nullable|exists:rooms,id',
            'area_name' => 'nullable|string|max:255'
        ]);
        $projectId = $request->project_id;
        // If no room selected → custom required
        if (!$request->room_id && !$request->area_name) {
            return response()->json([
                'success' => false,
                'message' => 'Custom room name required'
            ], 422);
        }

        // Fetch room name if selected
        $areaName = $request->area_name;

        if ($request->room_id) {
            $room = Room::find($request->room_id);
            $areaName = $room->name;
        }

        $sortOrder = ProjectArea::where('project_id', $projectId)->count() + 1;

        $area = ProjectArea::create([
            'project_id' => $projectId,
            'room_id' => $request->room_id,
            'area_name' => $areaName,
            'sort_order' => $sortOrder
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Area added successfully',
            'data' => $area
        ]);
    }

    // public function show(Request $request)
    // {
    //     $project = Project::with([
    //         'areas.items.product',
    //         'areas.items.variant',
    //         'areas.items.measurement',
    //         'areas.items.curtain',
    //         'areas.items.mattress',
    //         'areas.items.sofa',
    //         'areas.items.wallpaper',
    //         'areas.items.accessories'
    //     ])->findOrFail($request->project_id);

    //     return response()->json([
    //         'id' => $project->id,
    //         'name' => $project->project_name,
    //         'areas' => $project->areas->map(function ($area) {

    //             return [
    //                 'id' => $area->id,
    //                 'name' => $area->area_name,

    //                 'items' => $area->items->map(function ($item) {

    //                     return [
    //                         'id' => $item->id,
    //                         'category' => $item->category,
    //                         'product' => $item->product?->name,
    //                         'variant' => $item->variant?->sku,
    //                         'quantity' => $item->quantity,

    //                         'measurement' => $item->measurement ? [
    //                             'width' => $item->measurement->width,
    //                             'height' => $item->measurement->height,
    //                             'length' => $item->measurement->length,
    //                             'unit' => $item->measurement->unit ?? 'cm'
    //                         ] : null,

    //                         'curtain' => $item->curtain,
    //                         'mattress' => $item->mattress,
    //                         'sofa' => $item->sofa,
    //                         'wallpaper' => $item->wallpaper,
    //                         'accessories' => $item->accessories
    //                     ];
    //                 })
    //             ];
    //         })
    //     ]);
    // }

    public function show(Request $request)
    {
        $id = $request->project_id;
        $project = Project::with([
            'customer:id,name,phone',
            'dealer:id,name',
            'architect:id,name',
            'areas.room:id,name,image',

            'areas.items.product:id,name',
            'areas.items.variant:id,sku,offer_price',

            'areas.items.measurement',
            'areas.items.curtain',
            'areas.items.mattress',
            'areas.items.sofa',
            'areas.items.wallpaper',
            'areas.items.accessories'
        ])->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => [
                'project' => [
                    'id' => $project->id,
                    'project_number' => $project->project_number,
                    'project_name' => $project->project_name,
                    'status' => $project->status,
                    'project_type' => $project->project_type,

                    'customer' => [
                        'name' => $project->customer_name,
                        'phone' => $project->customer_phone
                    ],

                    'location' => [
                        'address' => $project->address,
                        'lat' => $project->location_lat,
                        'lng' => $project->location_lng
                    ],

                    'expected_completion' => $project->expected_completion_date,
                    'measurement_taken_by' => $project->employee_name,
                    'employee_contact' => $project->employee_contact
                ],

                'areas' => $project->areas->map(function ($area) {

                    return [
                        'id' => $area->id,
                        'room' => $area->room?->name,
                        'room_image' => asset($area->room?->image),
                        'area_name' => $area->area_name,

                        'items' => $area->items->map(function ($item) {

                            return [
                                'id' => $item->id,
                                'is_custom' => $item->is_custom,
                                'category' => $item->category,
                                'product' => $item->product?->name,
                                'variant' => $item->variant?->sku,
                                'price' => $item->variant?->offer_price,
                                'quantity' => $item->quantity,

                                'measurement' => $item->measurement,

                                'curtain' => $item->curtain,
                                'mattress' => $item->mattress,
                                'sofa' => $item->sofa,
                                'wallpaper' => $item->wallpaper,
                                'accessories' => $item->accessories,

                            ];
                        })
                    ];
                })
            ]
        ]);
    }

    private function formatMeasurement($item)
    {
        if (!$item->measurement) {
            return null;
        }

        return match ($item->type) {
            'curtain' =>
            'Width ' . $item->measurement->width . ' cm × Height ' . $item->measurement->height . ' cm',

            'mattress' =>
            $item->measurement->length . ' × ' . $item->measurement->width . ' in',

            default =>
            null
        };
    }

    public function index()
    {
        $projects = Project::with([
            'areas.room:id,name,image'
        ])
            ->latest()
            ->where('created_by',auth()->user()->id)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $projects->map(function ($project) {

                // Get first room image
                $roomImage = optional(
                    optional($project->areas->first())->room
                )->image;

                // Default image fallback
                $imagePath = $roomImage
                    ? asset($roomImage)
                    : asset('assets/images/room.jpg');

                return [
                    'id' => $project->id,
                    'project_number' => $project->project_number,
                    'project_name' => $project->project_name,
                    'customer_name' => $project->customer_name,
                    'address' => $project->address,
                    'expected_completion_date' => $project->expected_completion_date,
                    'room_image' => $imagePath
                ];
            })
        ]);
    }

    public function destroy($id)
    {
        $project = Project::findOrFail($id);

        $project->delete();

        return response()->json([
            'success' => true,
            'message' => 'Project deleted successfully'
        ]);
    }


    public function destroyArea($id)
    {
        $area = ProjectArea::findOrFail($id);

        $area->delete();

        return response()->json([
            'success' => true,
            'message' => 'Area deleted successfully'
        ]);
    }

    public function destroyItem($id)
    {
        $item = ProjectItem::findOrFail($id);

        $item->delete();

        return response()->json([
            'success' => true,
            'message' => 'Item deleted successfully'
        ]);
    }

    //WHEN CATEGORY SELECTED


    // public function getCategoryData($category_id)
    // {
    //     $category = Category::findOrFail($category_id);

    //     // MODELS
    //     $models = $category->models
    //         ? array_map('trim', explode(',', $category->models))
    //         : [];

    //     // SUBCATEGORIES
    //     $subcategories = SubCategory::whereIn(
    //         'id',
    //         Product::where('category_id', $category_id)
    //             ->whereNotNull('subcategory_id')
    //             ->pluck('subcategory_id')
    //     )
    //         ->select('id', 'name')
    //         ->get();

    //     // ✅ CATALOGUES WITH PRODUCT INFO
    //     $catId = ($category_id == 5) ? 4 : $category_id;

    // $catalogues = Product::where('category_id', $catId)
    // ->whereNotNull('master_catalogue_name')
    //  ->select(
    //                 'id as product_id',
    //                 'name as product_name',
    //                 DB::raw("master_catalogue_name as catalogue")
    //             )
    // ->distinct()
    // ->orderBy('master_catalogue_name')
    // ->get();

    //     if ($category_id == 5) {
    //         $catalogues = Product::where('category_id', 4)
    //             ->whereNotNull('master_catalogue_name')
    //             ->select(
    //                 'id as product_id',
    //                 'name as product_name',
    //                 DB::raw("master_catalogue_name as catalogue")
    //             )
    //             ->distinct()
    //             ->get();
    //     } else {
    //         $catalogues = Product::where('category_id', $category_id)
    //             ->whereNotNull('master_catalogue_name')
    //             ->select(
    //                 'id as product_id',
    //                 'name as product_name',
    //                 DB::raw("master_catalogue_name as catalogue")
    //             )
    //             ->distinct()
    //             ->get();
    //     }


    //     return response()->json([
    //         'status' => true,
    //         'models' => $models,
    //         'subcategories' => $subcategories,
    //         'catalogues' => $catalogues
    //     ]);
    // }

    public function getCategoryData($category_id)
    {
        $category = Category::findOrFail($category_id);

        // MODELS
        $models = $category->models
            ? array_map('trim', explode(',', $category->models))
            : [];

        // SUBCATEGORIES
        $subcategories = SubCategory::whereIn(
            'id',
            Product::where('category_id', $category_id)
                ->whereNotNull('subcategory_id')
                ->pluck('subcategory_id')
        )
            ->select('id', 'name')
            ->get();

        /*
    |--------------------------------------------------------------------------
    | CATALOGUES
    |--------------------------------------------------------------------------
    */

        if ($category->slug === 'mattress') {
            $catId = ($category_id == 5) ? 4 : $category_id;
            // ❌ NO DISTINCT for mattress
            $catalogues = Product::where('category_id', $category_id)
                ->whereNotNull('master_catalogue_name')
                ->select(
                    'id as product_id',
                    'name as product_name',
                    'master_catalogue_name as catalogue'
                )
                ->get();
        } elseif ($category->slug === 'curtain') {
            $catId = ($category_id == 5) ? 4 : $category_id;

            // ✅ DISTINCT catalogue for other categories
            $catalogues = Product::where('category_id', $catId)
                ->whereNotNull('master_catalogue_name')
                ->selectRaw('
                MIN(id) as product_id,
                MIN(name) as product_name,
                master_catalogue_name as catalogue
            ')
                ->groupBy('master_catalogue_name')
                ->get();
        } else {
            $catalogues = Product::where('category_id', $category_id)
                ->whereNotNull('master_catalogue_name')
                ->selectRaw('
                MIN(id) as product_id,
                MIN(name) as product_name,
                master_catalogue_name as catalogue
            ')
                ->groupBy('master_catalogue_name')
                ->get();
        }

        return response()->json([
            'status' => true,
            'models' => $models,
            'subcategories' => $subcategories,
            'catalogues' => $catalogues
        ]);
    }

    //SUBCATEGORY CHANGE
    // public function getSubcategoryCatalogues($subcategory_id)
    // {
    //     // Get subcategory
    //     $subcategory = SubCategory::findOrFail($subcategory_id);
    //     $models = $subcategory->models
    //         ? array_map('trim', explode(',', $subcategory->models))
    //         : [];
    //     // Get its category
    //     $category_id = $subcategory->category_id;

    //     // If Curtain category (5), use Fabric category (4)
    //     $catalogueCategoryId = ($category_id == 5) ? 4 : $category_id;

    //     $catalogues = Product::where('category_id', $catalogueCategoryId)
    //         ->whereNotNull('master_catalogue_name')
    //         ->select(
    //             'id as product_id',
    //             'name as product_name',
    //             DB::raw("master_catalogue_name as catalogue")
    //         )
    //         ->distinct()
    //         ->get();

    //     return response()->json([
    //         'status' => true,
    //         'models' => $models,
    //         'catalogues' => $catalogues
    //     ]);
    // }
public function getSubcategoryCatalogues($subcategory_id)
{
    $subcategory = SubCategory::findOrFail($subcategory_id);
    $category = Category::findOrFail($subcategory->category_id);

    $models = $subcategory->models
        ? array_map('trim', explode(',', $subcategory->models))
        : [];

    if ($category->slug === 'mattress') {

        $catalogues = Product::where('category_id', $category->id)
            ->where('subcategory_id', $subcategory_id)
            ->whereNotNull('master_catalogue_name')
            ->select(
                'id as product_id',
                'name as product_name',
                'master_catalogue_name as catalogue'
            )
            ->get();

    } else {

        $catalogueCategoryId = ($category->slug === 'curtain') ? 4 : $category->id;

        $catalogues = Product::where('category_id', $catalogueCategoryId)
            ->whereNotNull('master_catalogue_name')
            ->selectRaw('
                MIN(id) as product_id,
                MIN(name) as product_name,
                master_catalogue_name as catalogue
            ')
            ->groupBy('master_catalogue_name')
            ->orderBy('master_catalogue_name')
            ->get();
    }

    return response()->json([
        'status' => true,
        'models' => $models,
        'catalogues' => $catalogues
    ]);
}



    //CATALOGUE SELECTED → LOAD SKU
    public function getCatalogueSkus(Request $request)
    {
        $product_id = $request->product_id;

        // 1️⃣ Get product
        $product = Product::find($product_id);

        if (!$product) {
            return response()->json([
                'status' => false,
                'message' => 'Product not found'
            ]);
        }

        // 2️⃣ Get master catalogue name
        $masterCatalogueName = $product->master_catalogue_name;

        // 3️⃣ Get all product IDs having same master_catalogue_name
        $productIds = Product::where('master_catalogue_name', $masterCatalogueName)
            ->pluck('id');

        // 4️⃣ Get all variants of those products
        $variants = ProductVariant::whereIn('product_id', $productIds)->get();


        $skus = $variants->map(function ($v) {

            // ✅ SOFA DIMENSION FORMAT
            $sofa_dimension_label = null;

            if ($v->dimensions) {

                $dim = is_string($v->dimensions)
                    ? json_decode($v->dimensions, true)
                    : $v->dimensions;

                if (is_array($dim)) {

                    $length = $dim['length'] ?? null;
                    $width  = $dim['width'] ?? null;
                    $height = $dim['height'] ?? null;

                    // convert cm → inch if needed
                    if ($v->unit == 'cm') {
                        $length = $length ? round($length / 2.54) : null;
                        $width  = $width  ? round($width / 2.54)  : null;
                        $height = $height ? round($height / 2.54) : null;
                    }

                    if ($length && $width && $height) {
                        $sofa_dimension_label =
                            "L {$length} x W {$width} x H {$height}";
                    }
                }
            }

            return [
                'variant_id' => $v->id,
                'product_id' => $v->product_id,
                'sku' => $v->sku,
                // 'batch_no' => $v->batch_no,
                'mrp' => $v->mrp,
                'offer_price' => $v->offer_price,
                'unit' => $v->unit,
                // 'color_name' => $v->color_name,
                // 'color_code' => $v->color_code,
                // 'pattern' => $v->pattern,
                // 'material_or_fabric_composition' => $v->material_or_fabric_composition,
                // 'gsm' => $v->gsm,
                // 'weight' => $v->weight,
                // 'rub_count' => $v->rub_count,
                // 'moq' => $v->moq,
                // 'is_active' => $v->is_active,
                // 'fast_moving' => $v->fast_moving,
                // 'slow_moving' => $v->slow_moving,
                // 'design_repeat' => $v->design_repeat,
                // 'mattress_size_category' => $v->mattress_size_category,
                // 'mattress_length_inch' => $v->mattress_length_inch,
                // 'mattress_width_inch' => $v->mattress_width_inch,
                // 'mattress_thickness_inch' => $v->mattress_thickness_inch,
                'dimensions' => $sofa_dimension_label
            ];
        });
        // ✅ Mattress Config
        $mattress_config = [
            'categories' => [],
            'thickness'  => [],
            'units'      => ['inch', 'ft', 'cm'],
            'sizes'      => []
        ];

        foreach ($variants as $v) {

            if (
                !$v->mattress_size_category ||
                !$v->mattress_length_inch ||
                !$v->mattress_width_inch
            ) {
                continue;
            }

            $category = strtolower($v->mattress_size_category);

            // categories
            if (!in_array($category, $mattress_config['categories'])) {
                $mattress_config['categories'][] = $category;
            }

            // thickness
            if (
                $v->mattress_thickness_inch &&
                !in_array($v->mattress_thickness_inch, $mattress_config['thickness'])
            ) {
                $mattress_config['thickness'][] = $v->mattress_thickness_inch;
            }

            // inch
            $inch_length = $v->mattress_length_inch;
            $inch_width  = $v->mattress_width_inch;

            // ft
            $ft_length = round($inch_length / 12, 2);
            $ft_width  = round($inch_width / 12, 2);

            // cm
            $cm_length = round($inch_length * 2.54);
            $cm_width  = round($inch_width * 2.54);

            $mattress_config['sizes'][$category][] = [
                'variant_id' => $v->id,
                'inch' => "{$inch_length}x{$inch_width}",
                'ft'   => "{$ft_length}x{$ft_width}",
                'cm'   => "{$cm_length}x{$cm_width}",
            ];
        }


        return response()->json([
            'status' => true,
            'skus' => $skus,
            'mattress_config' => $mattress_config,

        ]);
    }

    public function calculate(Request $request)
    {
        if ($request->type == 'fabric') {
            return $this->calculateFabric($request);
        }

        if ($request->type == 'wallpaper') {
            return $this->calculateWallpaper($request);
        }

        return response()->json(['message' => 'Invalid type'], 400);
    }

    private function calculateFabric(Request $request)
    {
        $validated = $request->validate([
            'fabric_width'    => 'required|numeric|min:0.1',
            'price_per_meter' => 'required|numeric|min:1',
            'wall_width'      => 'required|numeric|min:0.1',
            'wall_height'     => 'required|numeric|min:0.1',
            'unit'            => 'required|in:inch,ft,cm,meter',
        ]);

        $fabricWidth = (float) $validated['fabric_width'];
        $wallWidth   = (float) $validated['wall_width'];
        $wallHeight  = (float) $validated['wall_height'];
        $price       = (float) $validated['price_per_meter'];
        $unit        = $validated['unit'];

        // Convert to meters
        switch ($unit) {
            case 'inch':
                $fabricWidth /= 39.37;
                $wallWidth   /= 39.37;
                $wallHeight  /= 39.37;
                break;

            case 'ft':
                $fabricWidth /= 3.281;
                $wallWidth   /= 3.281;
                $wallHeight  /= 3.281;
                break;

            case 'cm':
                $fabricWidth /= 100;
                $wallWidth   /= 100;
                $wallHeight  /= 100;
                break;
        }

        // 1️⃣ Panels required
        $panels = ceil($wallWidth / $fabricWidth);

        // 2️⃣ Total fabric required (meters)
        $fabricRequired = $panels * $wallHeight;

        // 3️⃣ Total amount
        $amount = $fabricRequired * $price;

        return response()->json([
            'success' => true,
            'data' => [
                'panels' => $panels,
                'fabric_required_m' => round($fabricRequired, 2),
                'price_per_meter' => $price,
                'amount' => round($amount, 2),
            ]
        ]);
    }


    private function calculateWallpaper(Request $request)
    {
        $validated = $request->validate([
            'roll_width'      => 'required|numeric|min:0.1',
            'roll_length'     => 'required|numeric|min:0.1',
            'price_per_roll'  => 'required|numeric|min:1',
            'wall_width'      => 'required|numeric|min:0.1',
            'wall_height'     => 'required|numeric|min:0.1',
            'design_repeat'   => 'nullable|numeric|min:0',
            'unit'            => 'required|in:inch,ft,cm,meter',
        ]);

        $rollWidth   = (float) $validated['roll_width'];
        $rollLength  = (float) $validated['roll_length'];
        $wallWidth   = (float) $validated['wall_width'];
        $wallHeight  = (float) $validated['wall_height'];
        $price       = (float) $validated['price_per_roll'];
        $repeat      = (float) ($validated['design_repeat'] ?? 0);
        $unit        = $validated['unit'];

        // Convert to meters
        switch ($unit) {
            case 'inch':
                $rollWidth  /= 39.37;
                $rollLength /= 39.37;
                $wallWidth  /= 39.37;
                $wallHeight /= 39.37;
                break;

            case 'ft':
                $rollWidth  /= 3.281;
                $rollLength /= 3.281;
                $wallWidth  /= 3.281;
                $wallHeight /= 3.281;
                break;

            case 'cm':
                $rollWidth  /= 100;
                $rollLength /= 100;
                $wallWidth  /= 100;
                $wallHeight /= 100;
                break;
        }

        // 1️⃣ Vertical cuttings required
        $cuttings = ceil($wallWidth / $rollWidth);

        // 2️⃣ Cutting height with repeat
        $cuttingHeight = $wallHeight + $repeat;

        // 3️⃣ Total length required
        $totalLength = $cuttings * $cuttingHeight;

        // 4️⃣ Rolls required
        $rolls = ceil($totalLength / $rollLength);

        // 5️⃣ Total amount
        $amount = $rolls * $price;

        return response()->json([
            'success' => true,
            'data' => [
                'cuttings' => $cuttings,
                'rolls_required' => $rolls,
                'amount' => round($amount, 2),
            ]
        ]);
    }



    public function getProjectItems(Request $request)
    {
        $itemId = $request->item_id;

        $items = ProjectItem::with([
            'measurement',
            'curtain',
            'mattress',
            'wallpaper',
            'product',
            'variant'
        ])
            ->where('id', $itemId)
            ->get();

        $data = $items->map(function ($item) {

            return [
                'id'          => $item->id,
                'project_id'  => $item->project_id,
                'area_id'     => $item->area_id,
                'category'    => $item->category,
                'product_id'  => $item->product_id,
                'variant_id'  => $item->variant_id,
                'product_name' => optional($item->product)->name,
                'variant_sku' => optional($item->variant)->sku,
                'quantity'    => $item->quantity,
                'notes'       => $item->notes,

                'measurement' => $item->measurement,

                'details' => match ($item->category) {
                    'curtain'   => $item->curtain,
                    'mattress'  => $item->mattress,
                    'wallpaper' => $item->wallpaper,
                    default     => null
                }
            ];
        });

        return response()->json([
            'success' => true,
            'data'    => $data
        ]);
    }

    public function storeCustomItem(Request $request)
    {
        $request->validate([
            'project_id' => 'required|exists:projects,id',
            'area_id'    => 'nullable|exists:project_areas,id',
            'category'   => 'required|in:curtain,wallpaper',
        ]);

        DB::beginTransaction();

        try {

            // -----------------------------
            // CREATE OR UPDATE PROJECT ITEM
            // -----------------------------
            if ($request->project_item_id) {

                $projectItem = ProjectItem::findOrFail($request->project_item_id);
                $projectItem->update([
                    'notes' => $request->notes,
                ]);
            } else {

                $projectItem = ProjectItem::create([
                    'project_id' => $request->project_id,
                    'area_id'    => $request->area_id,
                    'category'   => $request->category,
                    'is_custom'  => 1,
                    'notes'      => $request->notes,
                ]);
            }

            // -----------------------------
            // CURTAIN CUSTOM
            // -----------------------------
            if ($request->category == 'curtain') {

                ProjectItemCurtain::updateOrCreate(
                    ['project_item_id' => $projectItem->id],
                    [
                        'brand'           => $request->brand,
                        'catalogue'       => $request->catalogue,
                        'serial_no'       => $request->sku,
                        'model'           => $request->model,
                        'model_price'     => $request->model_price,
                        'fabric_width'    => $request->fabric_width,
                        'price_per_meter' => $request->price_per_meter,
                        'unit'            => $request->unit,
                    ]
                );

                ProjectItemMeasurement::updateOrCreate(
                    ['project_item_id' => $projectItem->id],
                    [
                        'width'           => $request->width,
                        'height'       => $request->height,
                        'measurement_type'       => 'curtain',
                        'unit'            => $request->unit,
                    ]
                );
            }

            // -----------------------------
            // WALLPAPER CUSTOM
            // -----------------------------
            if ($request->category == 'wallpaper') {

                $rollsRequired = null;

                // if ($request->wall_width && $request->roll_width) {
                //     $rollsRequired = ceil(
                //         ($request->wall_width * $request->wall_height) /
                //         ($request->roll_width * $request->roll_length)
                //     );
                // }

                ProjectItemWallpaper::updateOrCreate(
                    ['project_item_id' => $projectItem->id],
                    [
                        'brand'          => $request->brand,
                        'catalogue'      => $request->catalogue,
                        'serial_no'      => $request->sku,
                        'roll_width'     => $request->roll_width,
                        'roll_length'    => $request->roll_length,
                        'design_repeat'  => $request->design_repeat,
                        'price_per_roll' => $request->price_per_roll,
                        'wall_width'     => $request->wall_width,
                        'wall_height'    => $request->wall_height,
                        'rolls_required'  => $request->rolls_required,
                    ]
                );


                ProjectItemMeasurement::updateOrCreate(
                    ['project_item_id' => $projectItem->id],
                    [
                        'width'           => $request->wall_width,
                        'height'       => $request->wall_height,
                        'measurement_type'       => 'wallpaper',
                        'unit'            => 'm',
                    ]
                );
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => $request->project_item_id ? 'Custom item updated' : 'Custom item added',
                'project_item_id' => $projectItem->id
            ]);
        } catch (\Exception $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function getCustomItem($id)
    {
        $item = ProjectItem::with([
            'measurement',
            'curtain',
            'wallpaper',
            'area'
        ])
            ->where('is_custom', 1)
            ->find($id);

        if (!$item) {
            return response()->json([
                'success' => false,
                'message' => 'Custom item not found'
            ], 404);
        }

        $details = null;

        if ($item->category === 'curtain') {
            $details = $item->curtain;
        }

        if ($item->category === 'wallpaper') {
            $details = $item->wallpaper;
        }

        return response()->json([
            'success' => true,
            'data' => [
                'project_item_id' => $item->id,
                'project_id'      => $item->project_id,
                'area_id'         => $item->area_id,
                'area_name'       => optional($item->area)->area_name,
                'category'        => $item->category,
                'notes'           => $item->notes,

                'measurement' => $item->measurement ? [
                    'width'  => $item->measurement->width,
                    'height' => $item->measurement->height,
                    'length' => $item->measurement->length,
                    'unit'   => $item->measurement->unit,
                ] : null,

                'details' => $details
            ]
        ]);
    }

    /*
    |--------------------------------------------------------------------------
    | List Projects With Uploaded Quotation
    |--------------------------------------------------------------------------
    */
    public function myquotations()
    {
        $quotations = Quotation::with([
            'areas.room:id,name,image'
        ])
            ->where('status', 'Final')
            ->latest()
            ->get();

        return response()->json([
            'success' => true,
            'data' => $quotations->map(function ($quotation) {

                // Get first room image
                $roomImage = optional(
                    optional($quotation->areas->first())->room
                )->image;

                // Default image fallback
                $imagePath = $roomImage
                    ? asset($roomImage)
                    : asset('assets/images/room.jpg');

                return [
                    'id' => $quotation->id,
                    'project_number' => $quotation->quotation_number,
                    'project_name' => $quotation->project_name,
                    'customer_name' => $quotation->customer_name,
                    'address' => $quotation->address,
                    'expected_completion_date' => $quotation->expected_completion_date,
                    'room_image' => $imagePath,
                    "total_amount" => $quotation->grand_total
                ];
            })
        ]);

        // $quotations = quotation::whereNotNull('quotation_pdf')
        //     ->select('id', 'quotation_name', 'quotation_number', 'customer_name', 'quotation_pdf', 'created_at')
        //     ->latest()
        //     ->get();


        // return response()->json([
        //     'success' => true,
        //     'count'   => $quotations->count(),
        //     'data'    => $quotations
        // ]);
    }

    /*
    |--------------------------------------------------------------------------
    | Download Quotation PDF
    |--------------------------------------------------------------------------
    */
    public function download(Request $request)
    {
        $quotation = Quotation::with([
            'project.areas.items.measurement',
            'project.areas.items.product',
            'project.areas.items.variant.primaryImage',
            'items.details'
        ])->findOrFail($request->quotation_id);
        $groupedItems = $quotation->items->groupBy('area_id');
        $pdf = Pdf::loadView('pdf.cusella-quotation', compact('quotation', 'groupedItems'))
            ->setPaper('A4');

        return $pdf->download('quotation_' . $quotation->quotation_number . '.pdf');

        // $id = $request->project_id;
        // $project = Project::findOrFail($id);

        // if (!$project->quotation_pdf) {
        //     return response()->json([
        //         'success' => false,
        //         'message' => 'No quotation uploaded for this project'
        //     ], 404);
        // }

        // $filePath = public_path($project->quotation_pdf);

        // if (!File::exists($filePath)) {
        //     return response()->json([
        //         'success' => false,
        //         'message' => 'File not found'
        //     ], 404);
        // }

        // $fileName = str_replace(' ', '_', $project->project_name) . '_Quotation.pdf';

        // return response()->download($filePath, $fileName);
    }
}
