<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;
use Illuminate\Support\Facades\Cache;
use Illuminate\Http\Request;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;
    use HasApiTokens;
    use HasRoles;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'role_id',
        'user_type',
        'reporting_manager_id',
        'name',
        'email',
        'email_verified_at',
        'password',
        'phone',
        'address',
        'city',
        'state',
        'country',
        'pincode',
        'profile_image',
        'remember_token',
        'last_login_at',
        'status',
        'joining_date',
        'access_expires_at',
        'created_at',
        'updated_at',
        'last_activity_at',
    ];




    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'created_at',
        'updated_at',

    ];

    public function loginActivities()
    {
        return $this->hasMany(LoginActivity::class);
    }

    public function activityLog()
    {
        return $this->hasMany(ActivityLog::class);
    }
    public function isOnline(): bool
    {
        return Cache::has("user-is-online-{$this->id}");
    }
    // Department
    public function department()
    {
        return $this->belongsTo(Department::class, 'department_id');
    }
    public function orders()
    {
        return $this->hasMany(Order::class);
    }
    // Reporting Manager
    public function manager()
    {
        return $this->belongsTo(User::class, 'reporting_manager_id');
    }
    public function getProfileImageAttribute($value)
    {
        if (!$value) {
            return null;
        }

        // If already full URL (future S3/CDN support)
        if (filter_var($value, FILTER_VALIDATE_URL)) {
            return $value;
        }

        return url($value);
    }
    public function role()
    {
        return $this->belongsTo(Role::class);
    }

    public function logout(Request $request)
    {
        Cache::forget("user-is-online-" . auth()->id());

        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/');
    }

    // Assigned locations
    public function locations()
    {
        return $this->belongsToMany(
            Location::class,
            'user_locations'
        )->withTimestamps();
    }

    // Users assigned TO this user (manager → team)
    public function assignedUsers()
    {
        return $this->belongsToMany(
            User::class,
            'user_assignments',
            'user_id',
            'assigned_user_id'
        )->withTimestamps();
    }


    // Users who assigned THIS user (team → manager)
    public function assignedBy()
    {
        return $this->belongsToMany(
            User::class,
            'user_assignments',
            'assigned_user_id',
            'user_id'
        )->withPivot('department_id')
            ->withTimestamps();
    }


    // public function franchises()
    // {
    //     return $this->belongsToMany(
    //         Location::class,
    //         'user_franchises',
    //         'user_id',
    //         'franchise_id'
    //     )->withPivot('is_primary', 'assigned_by')->withTimestamps();
    // }

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }

    // App\Models\User.php
    // App\Models\User.php

    protected $appends = ['profile_image_url'];

    public function getProfileImageUrlAttribute()
    {
        if (!$this->profile_image) {
            return null;
        }

        // If DB has full URL, extract filename only
        $filename = basename($this->profile_image);

        return asset('assets/upload/users/' . $filename);
    }

    public function addresses()
    {
        return $this->hasMany(UserAddress::class);
    }

    public function defaultAddress()
    {
        return $this->hasOne(UserAddress::class)->where('is_default', true);
    }

    public function businessProfile()
{
    return $this->hasOne(BusinessProfile::class);
}

}
