<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\ProductReview;
use App\Models\Order;
use App\Models\SubCategory;
use App\Models\Product;
use App\Models\Room;
use Carbon\Carbon;
use App\Models\Cart;
use App\Models\Coupon;
use App\Models\ProductVariant;
use Illuminate\Support\Facades\File;
use App\Models\ProductReviewImage;

class ProductController extends Controller
{
    public function listing(Request $request)
    {
        $query = Product::where('is_active', 1)
            ->with([
                'variants' => function ($q) {
                    $q->where('is_active', 1)
                        ->with([
                            'images' => function ($q2) {
                                $q2->where('is_primary', 1);
                            }
                        ]);
                },
                'reviews'
            ]);

        // ---------------- SEARCH
        if ($request->filled('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // ---------------- CATEGORY
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        // ---------------- SUBCATEGORY
        if ($request->filled('subcategory_id')) {
            $query->where('subcategory_id', $request->subcategory_id);
        }

        // ---------------- ROOM
        if ($request->filled('room_id')) {
            $query->whereHas('rooms', function ($q) use ($request) {
                $q->where('rooms.id', $request->room_id);
            });
        }

        // ---------------- TAG
        if ($request->filled('tag_id')) {
            $query->whereHas('tags', function ($q) use ($request) {
                $q->where('tags.id', $request->tag_id);
            });
        }

        // ---------------- PRICE RANGE
        if ($request->filled('min_price') || $request->filled('max_price')) {
            $query->whereHas('variants', function ($q) use ($request) {
                if ($request->filled('min_price')) {
                    $q->where('offer_price', '>=', $request->min_price);
                }
                if ($request->filled('max_price')) {
                    $q->where('offer_price', '<=', $request->max_price);
                }
            });
        }

        // ---------------- RATING
        if ($request->filled('rating')) {
            $query->whereHas('reviews', function ($q) use ($request) {
                $q->havingRaw('AVG(rating) >= ?', [$request->rating]);
            });
        }

        // ---------------- SORT
        if ($request->filled('sort')) {
            match ($request->sort) {
                'popular'  => $query->withCount('reviews')->orderBy('reviews_count', 'desc'),
                'priority' => $query->orderBy('is_featured', 'desc'),
                default    => $query->latest(),
            };
        } else {
            $query->latest();
        }

        // ---------------- PAGINATION
        $perPage  = $request->input('per_page', 12);
        $products = $query->paginate($perPage);

        // ---------------- TRANSFORM (SAME AS productsByTag)
        $data = $products->getCollection()->map(function ($product) {

            $variant = $product->variants->first();
            $image   = optional($variant?->images->first())->image_path;

            return [
                'product_id'   => $product->id,
                'name'         => $product->name,
                'offer_price'  => $variant?->offer_price,
                'image'        => $image ? asset($image) : null,
                'rating'       => round($product->reviews->avg('rating'), 1),
                'review_count' => $product->reviews->count(),
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $data,
            'meta' => [
                'current_page' => $products->currentPage(),
                'last_page'    => $products->lastPage(),
                'per_page'     => $products->perPage(),
                'total'        => $products->total(),
            ]
        ]);
    }

    public function rooms()
    {
        $rooms = Room::select('id', 'name', 'image')
            ->orderBy('id')
            ->get()
            ->map(function ($room) {
                return [
                    'id'    => $room->id,
                    'name'  => $room->name,
                    'image' => $room->image ? asset($room->image) : null,
                ];
            });

        return response()->json([
            'success' => true,
            'data'    => $rooms
        ]);
    }

    public function categories()
    {
        $categories = Category::select(
            'id',
            'name',
            'image',
            'short_description'
        )
            ->orderBy('id')
            ->get()
            ->map(function ($category) {
                return [
                    'id'                => $category->id,
                    'name'              => $category->name,
                    'image'             => $category->image ? asset($category->image) : null,
                    'short_description' => $category->short_description,
                ];
            });

        return response()->json([
            'success' => true,
            'data'    => $categories
        ]);
    }
    public function subcategories(Request $request)
    {
        $categories = SubCategory::select(
            'id',
            'name',
            'image',
        )
            ->where('category_id', $request->category_id)
            ->orderBy('id')
            ->get()
            ->map(function ($category) {
                return [
                    'id'                => $category->id,
                    'name'              => $category->name,
                    'image'             => $category->image ? asset($category->image) : null,
                ];
            });

        return response()->json([
            'success' => true,
            'data'    => $categories
        ]);
    }
    public function show($id)
    {
        $product = Product::with([
            'category:id,name,slug',
            'subCategory:id,name',
            'rooms:id,name',
            'tags:id,name',
            'variants.images',
            'variants.stocks',
            'reviews.user:id,name'
        ])
            ->where('is_active', 1)
            ->findOrFail($id);

        $averageRating = round($product->reviews()->avg('rating'), 1);
        $reviewCount   = $product->reviews()->count();

        /*
    |--------------------------------------------------------------------------
    | Mattress Shared Images
    |--------------------------------------------------------------------------
    */

        $sharedMattressImages = collect();

        if ($product->category?->slug == 'mattress') {

            $variantWithImages = $product->variants
                ->first(fn($v) => $v->images->isNotEmpty());

            if ($variantWithImages) {

                $sharedMattressImages = $variantWithImages->images->map(function ($img) {
                    return [
                        'image'      => asset($img->image_path),
                        'is_primary' => $img->is_primary
                    ];
                })->values();
            }
        }

        /*
    |--------------------------------------------------------------------------
    | Recommendations
    |--------------------------------------------------------------------------
    */

        $recommendationIds = collect()
            ->merge($product->cross_sell_products ?? [])
            ->merge($product->related_products ?? [])
            ->merge($product->upsell_products ?? [])
            ->map(fn($id) => (int) $id)
            ->unique()
            ->filter()
            ->take(4)
            ->values();

        $recommendations = $recommendationIds->isNotEmpty()
            ? Product::whereIn('id', $recommendationIds)
            ->where('is_active', 1)
            ->with([
                'variants' => function ($q) {
                    $q->where('is_active', 1)
                        ->with(['images' => function ($q) {
                            $q->where('is_primary', 1);
                        }]);
                },
                'reviews'
            ])
            ->get()
            ->map(function ($product) {

                $variant = $product->variants->first();
                $image   = optional($variant?->images->first())->image_path;

                return [
                    'product_id'   => $product->id,
                    'name'         => $product->name,
                    'offer_price'  => $variant?->offer_price,
                    'image'        => $image ? asset($image) : null,
                    'rating'       => round($product->reviews->avg('rating'), 1),
                    'review_count' => $product->reviews->count(),
                ];
            })
            ->values()
            : [];

        /*
    |--------------------------------------------------------------------------
    | Mattress Config
    |--------------------------------------------------------------------------
    */

        $mattressConfig = [];

        if ($product->category?->slug == 'mattress') {

            $mattressConfig = [
                'categories' => $product->variants
                    ->pluck('mattress_size_category')
                    ->filter()
                    ->unique()
                    ->values(),

                'thickness' => $product->variants
                    ->pluck('mattress_thickness_inch')
                    ->filter()
                    ->unique()
                    ->values(),

                'units' => ['inch', 'ft', 'cm'],

                'sizes' => $product->variants
                    ->groupBy('mattress_size_category')
                    ->map(function ($variants) {

                        return $variants->map(function ($v) {

                            $l = round($v->mattress_length_inch);
                            $w = round($v->mattress_width_inch);

                            return [
                                'variant_id' => $v->id,
                                'inch' => "{$l}x{$w}",
                                'ft'   => round($l / 12, 2) . 'x' . round($w / 12, 2),
                                'cm'   => round($l * 2.54) . 'x' . round($w * 2.54),
                            ];
                        })->values();
                    })
            ];
        }

        /*
    |--------------------------------------------------------------------------
    | Response
    |--------------------------------------------------------------------------
    */

        return response()->json([
            'success' => true,
            'data' => [
                'product' => [
                    'id'                     => $product->id,
                    'name'                   => $product->name,
                    'master_catalogue'      => $product->master_catalogue_name. $product->master_catalogue_code,
                    'description'            => $product->description,
                    'material'               => $product->material,
                    'pattern'                => $product->pattern,
                    'features'               => $product->features,
                    'maintenance'            => $product->maintenance,
                    'warranty_certification' => $product->warranty_certification,
                    'measurement'            => $this->formatMeasurement($product->measurement),
                    'visibility'             => $product->visibility,
                    'gst_percent'            => $product->gst_percent,
                    'gst_inclusive'          => $product->gst_inclusive,

                    'category'    => $product->category,
                    'subcategory' => $product->subCategory,

                    'rooms' => $product->rooms->pluck('name'),
                    'tags'  => $product->tags->pluck('name'),

                    'average_rating' => $averageRating,
                    'review_count'   => $reviewCount,
                    'sample_purchase' => $product->sample_purchase,
                    'sample_purchase_rate' =>     $product->sample_purchase_rate
                ],
                'mattress_config' => $mattressConfig,

                'variants' => $product->variants
                    ->where('is_active', 1)
                    ->map(function ($variant) use ($product, $sharedMattressImages) {

                        $totalStock = $variant->stocks->sum('available_qty');

                        return [
                            'variant_id'  => $variant->id,
                            'sku'         => $variant->sku,
                            'batch_no'    => $variant->batch_no,
                            'color_name'  => $variant->color_name,
                            'color_code'  => $variant->color_code,
                            'material'    => $variant->material_or_fabric_composition,
                            'pattern'     => $variant->pattern,

                            'mrp'         => $variant->mrp,
                            'offer_price' => $variant->offer_price,
                            'unit'        => $variant->unit,
                            'moq'         => $variant->moq,

                            'dimensions'  => $variant->dimensions,

                            'images' => $variant->images->map(function ($img) {
                                return [
                                    'image'      => asset($img->image_path),
                                    'is_primary' => $img->is_primary
                                ];
                            }),

                            'stock' => [
                                'total_available_qty' => $totalStock,
                                'is_available'        => $totalStock > 0,
                            ],
                        ];
                    })
                    ->values(),
                'reviews' => $product->reviews->map(function ($review) {
                    return [
                        'user_name' => $review->user->name ?? 'Anonymous',
                        'rating'    => $review->rating,
                        'review'    => $review->review,
                        'date'      => $review->created_at->format('d M Y')
                    ];
                }),
                'recommendations' => $recommendations
            ]
        ]);
    }


    private function formatMeasurement($measurement)
    {
        if (!$measurement) {
            return null;
        }

        // Expecting "213 x 152 x 274"
        $parts = array_map('trim', explode('x', strtolower($measurement)));

        return [
            'length' => isset($parts[0]) ? (float) $parts[0] : null,
            'width'  => isset($parts[1]) ? (float) $parts[1] : null,
            'height' => isset($parts[2]) ? (float) $parts[2] : null,
        ];
    }

    public function storereview(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'order_id'   => 'required|exists:orders,id',
            'rating'     => 'required|integer|min:1|max:5',
            'review'     => 'nullable|string|max:1000',
            'images.*'   => 'image|mimes:jpg,jpeg,png,webp|max:2048'
        ]);
        $alreadyReviewed = ProductReview::where('product_id', $request->product_id)
            ->where('user_id', auth()->id())
            ->where('order_id', $request->order_id)
            ->exists();

        if ($alreadyReviewed) {
            return response()->json([
                'success' => false,
                'message' => 'You have already submitted a review for this product.',
            ], 409);
        }

        $order = Order::where('id', $request->order_id)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        $hasProduct = $order->items()
            ->where('product_id', $request->product_id)
            ->exists();

        if (!$hasProduct) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found in this order',
            ], 403);
        }

        $review = ProductReview::create([
            'product_id' => $request->product_id,
            'order_id'   => $request->order_id,
            'user_id'    => auth()->id(),
            'rating'     => $request->rating,
            'review'     => $request->review,
        ]);

        // ✅ Upload Multiple Images


        if ($request->hasFile('images')) {

            foreach ($request->file('images') as $image) {

                $imageName = time() . '-' . rand(100, 999) . '.' . $image->extension();

                $uploadPath = public_path('assets/upload/reviews/');

                if (!File::exists($uploadPath)) {
                    File::makeDirectory($uploadPath, 0755, true);
                }

                $image->move($uploadPath, $imageName);

                $imagePath = 'assets/upload/reviews/' . $imageName;

                ProductReviewImage::create([
                    'product_review_id' => $review->id,
                    'image' => $imagePath
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Review added successfully',
            //   'data' => $review->load('images'),
        ]);
    }


    public function productReviews(Product $product)
    {
        $reviews = ProductReview::with('images', 'user')
            ->where('product_id', $product->id)
            ->latest()
            ->get();

        $totalReviews = $reviews->count();

        // ⭐ Rating summary
        $ratingSummary = [];
        for ($i = 5; $i >= 1; $i--) {
            $ratingSummary[$i] = $reviews->where('rating', $i)->count();
        }

        $averageRating = $totalReviews > 0
            ? round($reviews->avg('rating'), 1)
            : 0;
        $reviewData = $reviews->map(function ($review) {

            return [
                'user_name'  => $review->user->name ?? null,
                'user_image' => $review->user->image ?? null,
                'rating'     => $review->rating,
                'review'     => $review->review,
                'time'       => $review->created_at->diffForHumans(),

                // ⭐ only new field added
                'images'     => $review->images->map(function ($img) {
                    return asset($img->image);
                })->values(),
            ];
        });

        return response()->json([
            'success' => true,
            'summary' => [
                'average_rating' => round($reviews->avg('rating')),
                'total_reviews'  => $reviews->count(),
                'rating_breakdown' => [
                    5 => $reviews->where('rating', 5)->count(),
                    4 => $reviews->where('rating', 4)->count(),
                    3 => $reviews->where('rating', 3)->count(),
                    2 => $reviews->where('rating', 2)->count(),
                    1 => $reviews->where('rating', 1)->count(),
                ],
            ],
            'reviews' => $reviewData
        ]);
    }

    // COUPON 



    public function couponList(Request $request)
    {
        $user = auth()->user();
        $today = now();

        $cart = Cart::with('items')->where('user_id', $user->id)->first();
        $cartTotal = $cart?->items->sum('subtotal') ?? 0;

        $coupons = Coupon::where('is_active', 1)
            ->where('start_date', '<=', $today)
            ->where(function ($q) use ($today) {
                $q->whereNull('end_date')
                    ->orWhere('end_date', '>=', $today);
            })
            ->where(function ($q) use ($cartTotal) {
                $q->whereNull('min_purchase')
                    ->orWhere('min_purchase', '<=', $cartTotal);
            })
            ->get();

        return response()->json([
            'success' => true,
            'cart_total' => $cartTotal,
            'user_role' => $user->role,
            'coupons_count' => $coupons->count(),
            'coupons' => $coupons,
        ]);
    }


    public function calculate(Request $request)
    {
        $product = Product::with('category')->findOrFail($request->product_id);
        $variant = ProductVariant::findOrFail($request->variant_id);

        switch ($product->category->slug) {

            case 'curtain':
                return $this->calculateCurtain($request, $variant);

            case 'wallpaper':
                return $this->calculateWallpaper($request, $variant);

            case 'fabric':
                return $this->calculateFabric($request, $variant);

            case 'mattress':
            case 'sofa':
            case 'accessories':
                return $this->calculateQuantity($request, $variant);

            default:
                return response()->json(['message' => 'Unknown category'], 400);
        }
    }

    private function calculateCurtain($request, $variant)
    {
        $width  = $request->width;
        $height = $request->height;

        $fabricWidth = $variant->gsm ?? 140;

        if ($request->type == 'roman') {

            $height += 30;
            $width  += 25;

            $panels = ceil($width / $fabricWidth);
        } else {

            $height += 30;
            $panels = ceil(($width * 2.5) / $fabricWidth);
        }

        $totalFabricCm = $height * $panels;
        $meter = $totalFabricCm / 100;

        $price = $meter * $variant->offer_price;

        return response()->json([
            'meter' => $meter,
            'panels' => $panels,
            'amount' => $price
        ]);
    }

    private function calculateWallpaper($request, $variant)
    {
        $request->validate([
            'width'  => 'required|numeric|min:0.1',
            'height' => 'required|numeric|min:0.1',
        ]);

        // decode dimensions
        $dim = is_array($variant->dimensions)
            ? $variant->dimensions
            : json_decode($variant->dimensions, true);

        // ✅ new dimension structure
        $rollWidth  = (float) ($dim['width'] ?? 0);   // roll width
        $rollLength = (float) ($dim['height'] ?? 0);  // roll height = roll length

        if ($rollWidth <= 0 || $rollLength <= 0) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid wallpaper roll dimensions'
            ], 422);
        }

        $wallWidth  = (float) $request->width;
        $wallHeight = (float) $request->height;

        $designRepeat = (float) ($variant->design_repeat ?? 0);

        // 1️⃣ number of cuttings
        $numberOfCuttings = ceil($wallWidth / $rollWidth);

        // 2️⃣ cutting height
        $cuttingHeight = $wallHeight + $designRepeat;

        // 3️⃣ total required length
        $totalRequiredLength = $cuttingHeight * $numberOfCuttings;

        // 4️⃣ rolls required
        $rollsRequired = ceil($totalRequiredLength / $rollLength);

        $amount = $rollsRequired * (float) $variant->offer_price;

        return response()->json([
            'success' => true,
            'data' => [
                'roll_width'  => $rollWidth,
                'roll_length' => $rollLength,
                'wall_width'  => $wallWidth,
                'wall_height' => $wallHeight,
                'design_repeat' => $designRepeat,

                'number_of_cuttings' => $numberOfCuttings,
                'cutting_height'     => round($cuttingHeight, 2),
                'total_length_m'     => round($totalRequiredLength, 2),
                'rolls_required'     => $rollsRequired,
                'price_per_roll'     => (float) $variant->offer_price,
                'amount'             => round($amount, 2),
            ]
        ]);
    }





    private function calculateFabric($request, $variant)
    {
        $request->validate([
            'meter' => 'required|numeric|min:0.1'
        ]);

        $meter = (float) $request->meter;
        $pricePerMeter = (float) $variant->offer_price;

        $totalPrice = round($meter * $pricePerMeter, 2);

        return response()->json([
            'success' => true,
            'data' => [
                'price_per_meter' => $pricePerMeter,
                'total_meters'    => $meter,
                'total_price'     => $totalPrice,
                'formatted' => [
                    'price_per_meter' => '₹' . number_format($pricePerMeter, 2),
                    'total_price'     => '₹' . number_format($totalPrice, 2),
                    'total_meters'    => number_format($meter, 2) . ' m'
                ]
            ]
        ]);
    }


    private function calculateQuantity($request, $variant)
    {
        $qty = $request->quantity ?? 1;

        return response()->json([
            'quantity' => $qty,
            'amount' => $qty * $variant->offer_price
        ]);
    }
}
