<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\Product;
use App\Models\WebSetting;
use Illuminate\Support\Facades\Auth;
use Razorpay\Api\Api;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\Cart;
use App\Models\OrderItem;
use App\Models\CartItem;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\CourierService;
use App\Services\FirebaseService;
use App\Notifications\SystemNotification;
use App\Models\User;

class OrderController extends Controller
{

    public function createOrder(Request $request)
    {
        $request->validate([
            'courier_id' => 'required|exists:courier_services,id',
            'billing_address_id' => 'required|exists:user_addresses,id',
            'shipping_address_id' => 'required|exists:user_addresses,id',
            'payment_method' => 'nullable|in:razorpay,cod'
        ]);

        $user = auth()->user();

        // 🔹 Get user's cart
        $cart = Cart::where('user_id', $user->id)->first();

        if (!$cart) {
            return response()->json([
                'success' => false,
                'message' => 'Cart not found'
            ], 400);
        }

        // 🔹 Get cart items with product and variant
        $cartItems = CartItem::where('cart_id', $cart->id)
            ->with(['product', 'variant']) // make sure CartItem has variant relation
            ->get();

        if ($cartItems->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'Cart is empty'
            ], 400);
        }

        DB::beginTransaction();

        try {
            $hasSample = $cartItems->contains('is_sample', 1);

            if ($hasSample) {
                $discount = 0;
                $couponId = null;
                $couponCode = null;
            }

            // 🔹 Totals
            $subTotal = $cartItems->sum('subtotal');
            $deliveryFee = 0;
            $discount = $cart->discount ?? 0;
            $couponId = $cart->coupon_id ?? null;
            $couponCode = optional($cart->coupon)->code ?? null;

            $grandTotal = $subTotal + $deliveryFee - $discount;


            // 🔹 Razorpay
            $setting = WebSetting::firstOrFail();

            $api = new Api(
                $setting->razropay_key_id,
                $setting->razropay_secret_key
            );

            $razorpayOrder = $api->order->create([
                'receipt' => 'ORD-' . time(),
                'amount' => $grandTotal * 100,
                'currency' => 'INR'
            ]);

            // 🔹 Create order
            $order = Order::create([
                'courier_id' => $request->courier_id,
                'user_id' => $user->id,
                'billing_address_id' => $request->billing_address_id,
                'shipping_address_id' => $request->shipping_address_id,
                'order_number' => 'ORD-' . time(),

                'sub_total' => $subTotal,
                'delivery_fee' => $deliveryFee,
                'discount' => $discount,
                'total_amount' => $grandTotal,

                'coupon_id' => $couponId,
                'coupon_code' => $couponCode,

                'payment_method' => $request->payment_method ?? 'razorpay',

                'razorpay_order_id' => $razorpayOrder['id'],
            ]);


            // 🔹 Create order items with variant_id
            foreach ($cartItems as $item) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $item->product_id,
                    'product_variant_id' => $item->variant_id ?? null, // ✅ added variant_id
                    'is_sample'          => $item->is_sample ?? 0, // ⭐ added
                    'is_fabric' =>  $item->is_fabric ?? 0,
                    'product_name' => $item->product->name,
                    'price' => $item->price,
                    'quantity' => $item->quantity,
                    'total' => $item->subtotal,
                ]);
            }


            DB::commit();

            return response()->json([
                'success' => true,
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'razorpay_order_id' => $razorpayOrder['id'],
                'razorpay_key' => $setting->razropay_key_id,
                'amount' => $grandTotal,
                'currency' => 'INR'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function placeOrder(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
            // 'razorpay_payment_id' => 'required',
            // 'razorpay_signature' => 'required',
        ]);

        $order = Order::where('id', $request->order_id)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        $setting = WebSetting::firstOrFail();

        // $api = new Api(
        //     $setting->razropay_key_id,
        //     $setting->razropay_secret_key
        // );

        // // 🔹 Verify signature
        // $api->utility->verifyPaymentSignature([
        //     'razorpay_order_id'   => $order->razorpay_order_id,
        //     'razorpay_payment_id' => $request->razorpay_payment_id,
        //     'razorpay_signature'  => $request->razorpay_signature,
        // ]);

        // $order->update([
        //     'payment_status' => 'success',
        //     'status' => 'paid',
        //     'razorpay_payment_id' => $request->razorpay_payment_id,
        //     'razorpay_signature' => $request->razorpay_signature,
        // ]);
        // 🔹 Clear cart
        $user = auth()->user();

        // 🔹 Get user's cart
        $cart = Cart::where('user_id', $user->id)->first();

        CartItem::where('cart_id', $cart->id)->delete();
        $cart->delete();
        //notification


        $title = 'Order Placed';
        $message = 'Your order #' . $order->order_number . ' placed successfully';
        $eventType = 'order_placed';
        $subjectId = $order->id;
        $subjectType = 'App\\Models\\Order';

        // ✅ DATABASE Notification
        $user->notify(new SystemNotification(
            $title,
            $message,
            $eventType,
            $subjectId,
            $subjectType
        ));


        // ✅ FIREBASE PUSH
        if (!empty($user->fcm_token)) {

            app(FirebaseService::class)->sendPush(
                $user->fcm_token,
                $title,
                $message,
                [
                    'event_type'   => $eventType,
                    'subject_id'   => (string)$subjectId,
                    'subject_type' => $subjectType,
                ]
            );
        }

        $admin = User::first();
        $adminId = $admin?->id;
        event(new \App\Events\SystemEvent([
            'type' => 'new_order',
            'title'        => 'New Role',
            'subject' => $user,
            'message' => "New order {$order->order_number} from {$user->name} ",
            'recipient_id' => $adminId, // Admin to notify
            'channels' => ['dashboard'],
        ]));

        return response()->json([
            'success' => true,
            'message' => 'Order placed successfully',
            'order_number' => $order->order_number
        ]);
    }


    public function myOrders(Request $request)
    {
        $user = Auth::user();
        $status = $request->query('status', 'active');

        $query = Order::with([
            'items.product:id,name',
            'items.product.variants.primaryImage'
        ])
            ->where('user_id', $user->id);

        switch ($status) {
            case 'active':
                $query->whereIn('status', ['placed', 'processing', 'shipped']);
                break;

            case 'completed':
                $query->where('status', 'delivered');
                break;

            case 'cancelled':
                $query->where('status', 'cancelled');
                break;

            default:
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid status'
                ], 422);
        }

        $orders = $query->latest()->get();

        // 🔄 Transform for UI
        $data = $orders->map(function ($order) {
            $firstItem = $order->items->first();
            $image = optional(
                $firstItem?->product?->variants?->first()?->primaryImage
            )->image_path;

            $image = $image ? asset($image) : null;

            return [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'image' => $image,
                'qty' => $order->items->sum('quantity'),
                'title' => $order->items->pluck('product_name')->implode(' | '),
                'amount' => $order->total_amount,
                'status' => $order->status,
                'created_at' => $order->created_at->format('d M Y'),
            ];
        });

        return response()->json([
            'success' => true,
            'status' => $status,
            'orders' => $data
        ]);
    }

    public function orderDetails($id)
    {
        $order = Order::with([
            'items.variant.primaryImage', // ✅ primary image for full width
            'billingAddress',
            'shippingAddress',
            'statusHistory',
        ])->findOrFail($id);

        // Transform items
        $items = $order->items->map(function ($item) {
            $imagePath = optional($item->variant?->primaryImage)->image_path;
            return [
                'product_id' => $item->product_id,
                'product_variant_id' => $item->product_variant_id,
                'name' => $item->product_name,
                'price' => (float) $item->price,
                'quantity' => (float) $item->quantity,
                'total' => (float) $item->total,
                'image' => $imagePath ? asset($imagePath) : null, // full URL for full-width display
            ];
        });

        // Transform status history
        $statusHistory = $order->statusHistory->map(function ($history) {
            return [
                'status' => $history->status,
                'note' => $history->note,
                'time' => Carbon::parse($history->created_at)->format('d M Y, h:i A'),
            ];
        });

        return response()->json([
            'success' => true,
            'order' => [
                'order_id' => $order->id,
                'order_number' => $order->order_number,
                'tracking_id' => $order->tracking_id,
                'courier' => $order->courier ? [
                    'id' => $order->courier->id,
                    'name' => $order->courier->name,
                    'website_address' => $order->courier->website,
                    'logo' => $order->courier->logo ? asset($order->courier->logo) : null,
                ] : null,
                'status' => $order->status,
                'order_date' => Carbon::parse($order->created_at)->format('d M Y'),
                'expected_delivery' => $order->expected_delivery_date
                    ? Carbon::parse($order->expected_delivery_date)->format('d M Y')
                    : null,
                'delivered_at' => $order->delivered_at
                    ? Carbon::parse($order->delivered_at)->format('d M Y h:i A')
                    : null,

                'amounts' => [
                    'sub_total' => (float) $order->sub_total,
                    'delivery_fee' => (float) $order->delivery_fee,
                    'discount' => (float) $order->discount,
                    'total' => (float) $order->total_amount,
                ],

                'payment' => [
                    'method' => $order->payment_method,
                    'status' => $order->payment_status,
                ],

                'items' => $items,

                'billing_address' => $order->billingAddress ? [
                    'name' => $order->billingAddress->name,
                    'address_line1' => $order->billingAddress->address_line1,
                    'address_line2' => $order->billingAddress->address_line2,
                    'city' => $order->billingAddress->city,
                    'state' => $order->billingAddress->state,
                    'country' => $order->billingAddress->country,
                    'pincode' => $order->billingAddress->pincode,
                    'phone' => $order->billingAddress->phone,
                    'gst_number' => $order->billingAddress->gst_number,
                ] : null,

                'shipping_address' => $order->shippingAddress ? [
                    'name' => $order->shippingAddress->name,
                    'address_line1' => $order->shippingAddress->address_line1,
                    'address_line2' => $order->shippingAddress->address_line2,
                    'city' => $order->shippingAddress->city,
                    'state' => $order->shippingAddress->state,
                    'country' => $order->shippingAddress->country,
                    'pincode' => $order->shippingAddress->pincode,
                    'phone' => $order->shippingAddress->phone,
                    'gst_number' => $order->shippingAddress->gst_number,
                ] : null,

                'status_history' => $statusHistory,
            ],
        ]);
    }

    public function courierlist()
    {
        $couriers = CourierService::select('id', 'name', 'logo')->get()->map(function ($courier) {
            $courier->logo = $courier->logo ? asset($courier->logo) : null;
            return $courier;
        });
        return response()->json([
            'success' => true,
            'data' => $couriers,
        ]);
    }

    public function cancelOrder(Request $request)
    {
        $request->validate([
            'reason' => 'required|string|max:500'
        ]);
        $id = $request->order_id;
        $order = Order::findOrFail($id);

        if ($order->status == 'cancelled') {
            return response()->json([
                'success' => false,
                'message' => 'Order already cancelled'
            ]);
        }

        if (in_array($order->status, ['completed'])) {
            return response()->json([
                'success' => false,
                'message' => 'Completed order cannot be cancelled'
            ]);
        }

        $order->update([
            'status' => 'cancelled',
            'cancel_reason' => $request->reason,
            'cancelled_at' => now()
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Order cancelled successfully'
        ]);
    }
}
