<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Cache;
use App\Models\Cart;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Carbon;
use App\Services\FirebaseService;
use App\Models\UserAddress;
use Kreait\Firebase\Exception\Auth\FailedToVerifyToken;
use App\Models\BusinessProfile;
use App\Notifications\SystemNotification;

class LoginController extends Controller
{

    public function testlogin(Request $request)
    {
        $request->validate([
            'phone' => 'required|string'
        ]);

        // ✅ Find user by phone
        $user = User::where('phone', $request->phone)->first();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found with this phone number'
            ], 404);
        }

        // ✅ Delete old tokens (optional, dev-friendly)
        $user->tokens()->delete();

        // ✅ Create sanctum token
        $token = $user->createToken('dev-login')->plainTextToken;

        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'token'   => $token,
            'user'    => [
                'id'    => $user->id,
                'name'  => $user->name,
                'phone' => $user->phone,
            ]
        ]);
    }
    public function login(Request $request, FirebaseService $firebase)
    {
        $request->validate([
            'firebase_token' => 'required|string',
        ]);

        try {
            $verifiedToken = $firebase->verifyIdToken($request->firebase_token);
            $firebaseUid = $verifiedToken->claims()->get('sub');
            $phone = $verifiedToken->claims()->get('phone_number');
        } catch (FailedToVerifyToken $e) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid Firebase token',
            ], 401);
        }

        $user = User::firstOrCreate(
            ['phone' => $phone],
            [
                'user_type' => 'customer',
                'status' => 'Active',
            ]
        );

        $user->tokens()->delete();

        $token = $user->createToken('mobile-login')->plainTextToken;

        $isProfileComplete = !empty($user->name) && !empty($user->email);

        $user->update(['last_login_at' => now()]);

        return response()->json([
            'status' => true,
            'message' => 'Login successful',
            'token' => $token,
            'is_new_user' => !$isProfileComplete,
            'next_screen' => $isProfileComplete ? 'home' : 'profile',
            'user' => [
                'id' => $user->id,
                'phone' => $user->phone,
                'user_type' => $user->user_type,
            ]
        ]);
    }

    public function updateProfile(Request $request)
    {
        $request->validate([
            'name'  => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . auth()->id()
        ]);

        $user = auth()->user();

        $user->update([
            'name'  => $request->name,
            'email' => $request->email,
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Profile updated successfully',
            'next_screen' => 'home'
        ]);
    }

    public function updatefullProfile(Request $request)
    {
        $request->validate([
            'name'  => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . auth()->id(),
            'phone' => 'required|string|max:255',
            'photo'      => 'nullable|image|mimes:jpg,jpeg,png|max:4096',
        ]);

        $user = auth()->user();

        $user->update([
            'name'  => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
        ]);
        $image = '';

        if ($request->photo) {
            $rand = rand(100, 999);
            $image = time() . '.' . $request->photo->extension();
            $imageOriginalName = $request->photo->getClientOriginalName();
            $request->photo->move(public_path('assets/upload/users/'), $image);
            User::where('id', auth()->user()->id)
                ->update([
                    'profile_image' => $image,
                ]);
        }
        return response()->json([
            'status' => true,
            'message' => 'Profile updated successfully',
        ]);
    }

    public function logout(Request $request)
    {
        // Delete only the current token
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'status' => true,
            'message' => 'Logged out successfully'
        ]);
    }

    // ********************* user address start  *************************
    public function index()
    {
        $user = Auth::user();
        return response()->json([
            'success' => true,
            'addresses' => $user->addresses
        ]);
    }
    public function defaultaddress()
    {
        $user = Auth::user();
        return response()->json([
            'success' => true,
            'addresses' => $user->defaultAddress
        ]);
    }
    public function store(Request $request)
    {
        $request->validate([
            'address_line1' => 'required|string|max:255',
            'city' => 'required|string|max:255',
            'is_default' => 'nullable|boolean',
        ]);

        $data = $request->only([
            'name',
            'address_line1',
            'address_line2',
            'city',
            'state',
            'country',
            'pincode',
            'phone',
            'is_default',
            'type_of_address',
            'gst_number',
            'alternative_phone'
        ]);

        $data['user_id'] = Auth::id();

        // ✅ convert true/false to 1/0
        $data['is_default'] = !empty($data['is_default']) ? 1 : 0;

        // ✅ If new address is default, unset previous default
        if ($data['is_default'] === 1) {
            UserAddress::where('user_id', Auth::id())
                ->update(['is_default' => 0]);
        }

        $address = UserAddress::create($data);

        return response()->json([
            'success' => true,
            'address' => $address
        ]);
    }

    public function update(Request $request, $id)
    {
        $address = UserAddress::where('user_id', Auth::id())
            ->findOrFail($id);

        $request->validate([
            'is_default' => 'nullable|boolean',
        ]);

        $data = $request->only([
            'name',
            'address_line1',
            'address_line2',
            'city',
            'state',
            'country',
            'pincode',
            //  'phone',
            'is_default',
            'type_of_address',
            'gst_number',
            'alternative_phone'
        ]);

        // ✅ Convert boolean → integer (ONLY if present)
        if ($request->has('is_default')) {
            $data['is_default'] = $request->boolean('is_default') ? 1 : 0;

            // ✅ If setting this as default → unset others
            if ($data['is_default'] === 1) {
                UserAddress::where('user_id', Auth::id())
                    ->where('id', '!=', $address->id)
                    ->update(['is_default' => 0]);
            }
        }

        $address->update($data);

        return response()->json([
            'success' => true,
            'address' => $address->fresh()
        ]);
    }

    public function setasDefaultaddress(Request $request)
    {
        $id = $request->address_id;
        $address = UserAddress::where('user_id', Auth::id())
            ->findOrFail($id);

        if ($address == null) {
            return response()->json([
                'success' => false,
                'message' => "Address Does not exist"
            ]);
        }

        $existingdefaultaddress = UserAddress::where('user_id', Auth::id())
            ->where('id', '!=', $address->id)->first();

        UserAddress::where('user_id', Auth::id())
            ->where('id', $existingdefaultaddress->id)
            ->update(['is_default' => 0]);
        // ✅ Convert boolean → integer (ONLY if present)


        // ✅ If setting this as default → unset others

        UserAddress::where('user_id', Auth::id())
            ->where('id', $address->id)
            ->update(['is_default' => 1]);



        return response()->json([
            'success' => true,
            'message' => "Address set as default address"
        ]);
    }

    public function destroy($id)
    {
        $address = UserAddress::where('user_id', Auth::id())->findOrFail($id);
        $address->delete();

        return response()->json(['success' => true, 'message' => 'Address deleted']);
    }

    // ********************* user address end *************************

    public function upgradeAccount(Request $request)
    {
        $request->validate([
            'role' => 'required|in:retail_firm,construction_firm',
            'legal_shop_name' => 'required',
            'legal_shop_address' => 'nullable',
            'shop_mobile_number' => 'nullable',
            'gst_number' => 'nullable'
        ]);

        $user = Auth::user();
        // echo $user->id;
        // print_r([ 'user_id'=>$user->id, // 👈 REQUIRED
        //         'business_type'=>$request->role,
        //         'legal_shop_name'=>$request->legal_shop_name,
        //         'legal_shop_address'=>$request->legal_shop_address,
        //         'shop_mobile_number'=>$request->shop_mobile_number,
        //         'gst_number'=>$request->gst_number]);die();
        // die();
        // update user_type
        // if($request->role == 'retail_firm'){
        //     $user->update(['user_type'=>'dealer']);
        // }else{
        //     $user->update(['user_type'=>'architect']);
        // }


        BusinessProfile::updateOrCreate(
            ['user_id' => $user->id],
            [
                'user_id' => $user->id, // 👈 REQUIRED
                'business_type' => $request->role,
                'legal_shop_name' => $request->legal_shop_name,
                'legal_shop_address' => $request->legal_shop_address,
                'shop_mobile_number' => $request->shop_mobile_number,
                'gst_number' => $request->gst_number
            ]
        );


        // 🔔 Admin Notification
     
            $admin = User::first();
            $adminId = $admin?->id;
            event(new \App\Events\SystemEvent([
                'type' => 'role_updated',
                'title'        => 'Role Change Request',
                'subject' => $user,
                'message' => "New role change request from {$user->name} ",
                'recipient_id' => $adminId, // Admin to notify
                'channels' => ['dashboard'],
            ]));


        return response()->json([
            'success' => true,
            'message' => 'Account upgrade request submitted'
        ]);
    }
}
