<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\Product;
use App\Models\Banner;
use App\Models\HomeSection;
use App\Models\Brands;
use Carbon\Carbon;
use App\Models\ProductVariant;
use Illuminate\Support\Facades\DB;
use App\Models\HomeVideos;
use App\Models\Room;
use App\Models\Tags;

class HomeController extends Controller
{
    public function banners(Request $request)
    {
        $today = now()->toDateString();

        $placement = $request->get('placement', 'home');
        $audience  = $request->get('audience', 'all');
        $platform  = $request->get('platform', 'all');

        $banners = Banner::select([
            'id',
            'title',
            'image',
            'banner_type',
            'placement',
            'audience',
            'platform',
            'priority',
        ])
            ->where('status', true)
            ->where('placement', $placement)

            // ✅ Audience filter
            ->where(function ($q) use ($audience) {
                if ($audience !== 'all') {
                    $q->whereIn('audience', [$audience, 'all']);
                }
            })

            // ✅ Platform filter (FIXED)
            ->where(function ($q) use ($platform) {
                if ($platform !== 'all') {
                    $q->whereIn('platform', [$platform, 'all']);
                }
            })

            // ✅ Date filter
            ->where(function ($q) use ($today) {
                $q->whereNull('start_date')
                    ->orWhereDate('start_date', '<=', $today);
            })
            ->where(function ($q) use ($today) {
                $q->whereNull('end_date')
                    ->orWhereDate('end_date', '>=', $today);
            })

            // ✅ Sorting
            ->orderBy('priority')
            ->orderBy('id')

            ->get();

        return response()->json([
            'status' => true,
            'data'   => $banners
        ]);
    }

    public function categories()
    {
        $categories = Category::select('id', 'name', 'icon')
            ->orderBy('id')
            ->get();

        return response()->json([
            'status' => true,
            'data' => $categories
        ]);
    }

    public function bannerView($id)
    {
        $banner = Banner::where('status', true)->find($id);

        if (!$banner) {
            return response()->json([
                'status' => false,
                'message' => 'Banner not found'
            ], 404);
        }

        $banner->increment('views');

        return response()->json([
            'status' => true,
            'message' => 'Banner view recorded'
        ]);
    }

    public function bannerClick($id)
    {
        $banner = Banner::where('status', true)->findOrFail($id);

        $banner->increment('clicks');

        return response()->json([
            'status'  => true,
            'message' => 'Banner click recorded'
        ]);
    }

    public function tags()
    {
        $tags = Tags::select('id', 'name')->get();

        return response()->json([
            'status' => true,
            'data' => $tags
        ]);
    }



    public function productsByTag($tagId = null)
    {
        $productsQuery = Product::query()
            ->where('is_active', 1)
            ->with([
                'variants' => function ($q) {
                    $q->where('is_active', 1)
                        ->with([
                            'images' => function ($q) {
                                $q->where('is_primary', 1);
                            }
                        ]);
                },
                'reviews'
            ]);

        // ✅ Apply tag filter ONLY if tagId is present
        if (!empty($tagId)) {
            $productsQuery->whereHas('tags', function ($q) use ($tagId) {
                $q->where('tags.id', $tagId);
            });
        }

        $products = $productsQuery
            ->limit(4)
            ->get();

        $data = $products->map(function ($product) {

            $variant = $product->variants->first();
            $image   = optional($variant?->images->first())->image_path;

            return [
                'product_id'   => $product->id,
                'name'         => $product->name,
                'offer_price'  => $variant?->offer_price,
                'image'        => $image ? asset($image) : null,
                'rating'       => round($product->reviews->avg('rating'), 1),
                'review_count' => $product->reviews->count(),
            ];
        })->values();

        return response()->json([
            'success' => true,
            'data'    => $data
        ]);
    }
    public function productsByTagFull(Request $request, $tagId = null)
    {
        $query = Product::where('is_active', 1)
            ->with([
                'variants' => function ($q) {
                    $q->where('is_active', 1)
                        ->with([
                            'images' => function ($q2) {
                                $q2->where('is_primary', 1);
                            }
                        ]);
                },
                'reviews'
            ]);

        // ✅ Apply tag filter ONLY if tagId is present
        if (!empty($tagId)) {
            $query->whereHas('tags', function ($q) use ($tagId) {
                $q->where('tags.id', $tagId);
            });
        }

        // ---------------- PAGINATION
        $perPage  = $request->input('per_page', 12);
        $products = $query->latest()->paginate($perPage);

        // ---------------- TRANSFORM
        $data = $products->getCollection()->map(function ($product) {

            $variant = $product->variants->first();
            $image   = optional($variant?->images->first())->image_path;

            return [
                'product_id'   => $product->id,
                'name'         => $product->name,
                'offer_price'  => $variant?->offer_price,
                'image'        => $image ? asset($image) : null,
                'rating'       => round($product->reviews->avg('rating'), 1),
                'review_count' => $product->reviews->count(),
            ];
        });

        return response()->json([
            'success' => true,
            'data'    => $data,
            'meta'    => [
                'current_page' => $products->currentPage(),
                'last_page'    => $products->lastPage(),
                'per_page'     => $products->perPage(),
                'total'        => $products->total(),
            ]
        ]);
    }



    public function bestSellerCategories()
    {
        $categories = Category::whereHas('products', function ($q) {
            $q->where('is_active', 1);
        })
            ->select('id', 'name')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $categories
        ]);
    }

    public function bestSellerProducts($category_id)
    {

        $products = Product::where('category_id', $category_id)
            ->where('is_active', 1)
            ->with([
                'variants' => function ($q) {
                    $q->where('is_active', 1)
                        ->with([
                            'images' => function ($q) {
                                $q->where('is_primary', 1);
                            }
                        ]);
                }
            ])
            ->limit(4)
            ->get();

        $data = $products->map(function ($product) {

            $variant = $product->variants->first();
            $image   = optional($variant?->images->first())->image_path;

            return [
                'product_id'   => $product->id,
                'name'         => $product->name,
                'offer_price'  => $variant?->offer_price,
                'image'        => $image ? asset($image) : null
            ];
        })->values();

        return response()->json([
            'success' => true,
            'data'    => $data
        ]);
    }

    public function specialForYou()
    {
        $products = Product::where('is_featured', 1)
            ->where('is_active', 1)
            ->with([
                'variants' => function ($q) {
                    $q->where('is_active', 1)
                        ->with([
                            'images' => function ($q) {
                                $q->where('is_primary', 1);
                            }
                        ]);
                },
                'reviews'
            ])
            ->limit(3)
            ->get();

        $data = $products->map(function ($product) {

            $variant = $product->variants->first();
            $image   = optional($variant?->images->first())->image_path;

            return [
                'product_id'   => $product->id,
                'name'         => $product->name,
                'offer_price'  => $variant?->offer_price,
                'image'        => $image ? asset($image) : null,
                'rating'       => round($product->reviews->avg('rating'), 1),
                'review_count' => $product->reviews->count(),
            ];
        })->values();

        return response()->json([
            'success' => true,
            'data'    => $data
        ]);
    }

    public function specialForYouFull(Request $request)
    {
        $query = Product::where('is_featured', 1)
            ->where('is_active', 1)
            ->with([
                'variants' => function ($q) {
                    $q->where('is_active', 1)
                        ->with([
                            'images' => function ($q2) {
                                $q2->where('is_primary', 1);
                            }
                        ]);
                },
                'reviews'
            ])
            ->latest();

        // ---------------- PAGINATION (same standard everywhere)
        $perPage  = $request->input('per_page', 12);
        $products = $query->paginate($perPage);

        // ---------------- TRANSFORM
        $data = $products->getCollection()->map(function ($product) {

            $variant = $product->variants->first();
            $image   = optional($variant?->images->first())->image_path;

            return [
                'product_id'   => $product->id,
                'name'         => $product->name,
                'offer_price'  => $variant?->offer_price,
                'image'        => $image ? asset($image) : null,
                'rating'       => round($product->reviews->avg('rating'), 1),
                'review_count' => $product->reviews->count(),
            ];
        });

        return response()->json([
            'success' => true,
            'data'    => $data,
            'meta'    => [
                'current_page' => $products->currentPage(),
                'last_page'    => $products->lastPage(),
                'per_page'     => $products->perPage(),
                'total'        => $products->total(),
            ]
        ]);
    }

    // {
    //     $products = Product::with([
    //             'variants.images' => fn($q) => $q->where('is_primary', 1)
    //         ])
    //         ->where('is_featured', 1)
    //         ->where('is_active', 1)
    //         ->orderByDesc('average_rating')
    //         ->limit(2)
    //         ->get();

    //     return response()->json([
    //         'status' => true,
    //         'data' => $products
    //     ]);
    // }

    public function exploreLuxuryCurtains()
    {
        $category = Category::where('slug', 'curtain')->first();

        if (!$category) {
            return response()->json([
                'success' => false,
                'data' => []
            ]);
        }

        $subCategories = $category->subCategories()
            ->where('is_active', 1)
            ->get()
            ->map(function ($sub) {
                return [
                    'id'    => $sub->id,
                    'name'  => $sub->name,
                    'image' => $sub->image_full, // already full URL
                ];
            })
            ->values();

        return response()->json([
            'success' => true,
            'data' => $subCategories
        ]);
    }
    public function brands()
    {
        $brands = Brands::select('id', 'name', 'slug', 'image')
            ->orderBy('name')
            ->get();

        return response()->json([
            'status' => true,
            'data' => $brands
        ]);
    }



    public function search(Request $request)
    {
        $search = trim($request->q);

        if (!$search) {
            return response()->json([
                'success' => true,
                'data' => []
            ]);
        }

        $products = Product::where('is_active', 1)
            ->where('name', 'like', "%{$search}%")
            ->select('id', 'name')
            ->limit(10)
            ->get()
            ->map(function ($product) {
                return [
                    'product_id'   => $product->id,
                    'product_name' => $product->name,
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $products
        ]);
    }

    public function homeVideo()
    {
        $video = HomeVideos::with('product')
            ->where('is_active', 1)
            ->first();

        if (!$video) {
            return response()->json([
                'status' => false,
                'message' => 'No home video found'
            ]);
        }

        return response()->json([
            'status' => true,
            'data' => [
                'title' => $video->title,
                'subtitle' => $video->subtitle,
                'button_text' => $video->button_text,
                'video' => asset($video->video),
                'product' => [
                    'id' => $video->product->id,
                    'name' => $video->product->name
                ]
            ]
        ]);
    }

    public function homeExplore()
    {
        $section = HomeSection::with([
            'category.subcategories' => function ($q) {
                $q->take(3);
            }
        ])->first();

        if (!$section) {
            return response()->json([
                'status' => false,
                'message' => 'No explore section found'
            ]);
        }

        $subcategories = $section->category->subcategories->map(function ($sub) {
            return [
                'id' => $sub->id,
                'name' => $sub->name,
                'image' => asset($sub->image)
            ];
        });

        return response()->json([
            'status' => true,
            'data' => [
                'title' => $section->title,
                'subtitle' => $section->subtitle,
                'category' => [
                    'id' => $section->category->id,
                    'name' => $section->category->name
                ],
                'subcategories' => $subcategories
            ]
        ]);
    }
}
