<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Cart;
use App\Models\ProductVariant;
use App\Models\CartItem;
use Illuminate\Support\Facades\Auth;
use App\Models\Coupon;

class CartController extends Controller
{
    // public function addToCart(Request $request)
    // {
    //     $userId = auth()->id();

    //     $request->validate([
    //         'product_id' => 'required|exists:products,id',
    //         'variant_id' => 'required|exists:product_variants,id',
    //         'quantity'   => 'required|integer|min:1',
    //     ]);

    //     $variant = ProductVariant::where('id', $request->variant_id)
    //         ->where('product_id', $request->product_id)
    //         ->with('stocks') // ✅ load all stocks
    //         ->firstOrFail();

    //     // ✅ Calculate usable stock across locations
    //     $availableQty = $variant->stocks
    //         ->where('is_available', 1)
    //         ->sum(function ($stock) {
    //             return max(
    //                 ($stock->available_qty ?? 0) - ($stock->reserved_qty ?? 0),
    //                 0
    //             );
    //         });

    //     if ($availableQty <= 0) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Product out of stock'
    //         ], 400);
    //     }

    //     if ($request->quantity > $availableQty) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Quantity exceeds available stock'
    //         ], 400);
    //     }

    //     $cart = Cart::firstOrCreate([
    //         'user_id' => $userId
    //     ]);

    //     $cartItem = CartItem::where([
    //         'cart_id'    => $cart->id,
    //         'variant_id' => $variant->id
    //     ])->first();

    //     if ($cartItem) {
    //         $cartItem->quantity += $request->quantity;
    //     } else {
    //         $cartItem = new CartItem();
    //         $cartItem->cart_id    = $cart->id;
    //         $cartItem->product_id = $request->product_id;
    //         $cartItem->variant_id = $variant->id;
    //         $cartItem->price      = $variant->offer_price;
    //         $cartItem->quantity   = $request->quantity;
    //     }

    //     // ✅ Re-check after increment
    //     if ($cartItem->quantity > $availableQty) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Quantity exceeds available stock'
    //         ], 400);
    //     }

    //     $cartItem->subtotal = $cartItem->price * $cartItem->quantity;
    //     $cartItem->save();

    //     return response()->json([
    //         'success'     => true,
    //         'message'     => 'Product added to cart',
    //         'cart_count'  => $cart->items()->sum('quantity')
    //     ]);
    // }

    // public function updateQuantity(Request $request)
    // {
    //     $request->validate([
    //         'cart_item_id' => 'required|exists:cart_items,id',
    //         'quantity'     => 'required|integer|min:1'
    //     ]);

    //     $cartItem = CartItem::where('id', $request->cart_item_id)
    //         ->whereHas(
    //             'cart',
    //             fn($q) =>
    //             $q->where('user_id', auth()->id())
    //         )
    //         ->with('variant.stocks')
    //         ->firstOrFail();

    //     $variant = $cartItem->variant;

    //     // ✅ Sum usable stock across all locations
    //     $availableQty = $variant->stocks
    //         ->where('is_available', 1)
    //         ->sum(function ($stock) {
    //             return max(
    //                 ($stock->available_qty ?? 0) - ($stock->reserved_qty ?? 0),
    //                 0
    //             );
    //         });

    //     if ($availableQty <= 0) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Product out of stock'
    //         ], 400);
    //     }

    //     if ($request->quantity > $availableQty) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Quantity exceeds available stock'
    //         ], 400);
    //     }

    //     $cartItem->quantity = $request->quantity;
    //     $cartItem->save();

    //     return response()->json([
    //         'success'  => true,
    //         'message'  => 'Quantity updated',
    //         'subtotal' => (float) $cartItem->subtotal
    //     ]);
    // }
    public function addToCart(Request $request)
    {

        $userId = auth()->id();

        /*
|--------------------------------------------------------------------------
| STEP 1 — Validate Required IDs
|--------------------------------------------------------------------------
*/
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'variant_id' => 'required|exists:product_variants,id',
            'is_sample'  => 'nullable|boolean',
            'is_fabric'  => 'nullable|boolean'
        ]);

        /*
|--------------------------------------------------------------------------
| STEP 2 — Load Product
|--------------------------------------------------------------------------
*/
        $variant = ProductVariant::with('product.category')
            ->where('id', $request->variant_id)
            ->where('product_id', $request->product_id)
            ->firstOrFail();

        $product = $variant->product;

        /*
|--------------------------------------------------------------------------
| STEP 3 — Detect Fabric
|--------------------------------------------------------------------------
*/
        $isFabricCategory = strtolower($product->category?->slug ?? '') == 'fabric';
        $isFabricRequest  = $request->is_fabric ?? 0;

        /*
|--------------------------------------------------------------------------
| STEP 4 — Quantity Validation (ONLY ONCE)
|--------------------------------------------------------------------------
*/
        $rules = [];

        if ($isFabricCategory && $isFabricRequest) {
            $rules['quantity'] = 'required|numeric|min:0.01';
        } else {
            $rules['quantity'] = 'required|integer|min:1';
        }

        $request->validate($rules);


        $variant = ProductVariant::with('product')
            ->where('id', $request->variant_id)
            ->where('product_id', $request->product_id)
            ->firstOrFail();

        $product = $variant->product;


        $cart = Cart::firstOrCreate([
            'user_id' => $userId
        ]);
        $isSample = $request->is_sample ?? 0;

        $cartItem = CartItem::where([
            'cart_id'    => $cart->id,
            'variant_id' => $variant->id,
            //  'is_sample'  => $isSample
        ])->first();


        if ($isSample && $product->sample_purchase) {
            $price = $product->sample_purchase_rate;
        } else {
            $price = $variant->offer_price;
        }
        $isFabric = strtolower($product->category?->slug ?? '') == 'fabric';
        if ($cartItem) {



            $cartItem->quantity += $request->quantity;
            $cartItem->price     = $price;
            $cartItem->subtotal  = $cartItem->quantity * $price;
        } else {

            $cartItem = new CartItem();
            $cartItem->cart_id    = $cart->id;
            $cartItem->product_id = $product->id;
            $cartItem->variant_id = $variant->id;
            $cartItem->is_sample  = $isSample;
            $cartItem->is_fabric  = $isFabricRequest;

           

                $cartItem->quantity = $request->quantity;
                $cartItem->price    = $price;
                $cartItem->subtotal = $price * $request->quantity;
            
        }

        $cartItem->save();

        return response()->json([
            'success'    => true,
            'message'    => 'Product added to cart',
           'cart_count' => $cart->items->sum(function ($item) {
    return $item->is_fabric == 1
        ? 1
        : $item->quantity;
}),

        ]);
    }
  

    public function updateQuantity(Request $request)
{
    $request->validate([
        'cart_item_id' => 'required|exists:cart_items,id',
        'quantity'     => 'required|numeric|min:0.1' // ✅ allow decimal
    ]);

    $cartItem = CartItem::with(['product']) // load product if is_fabric there
        ->where('id', $request->cart_item_id)
        ->whereHas('cart', function ($q) {
            $q->where('user_id', auth()->id());
        })
        ->firstOrFail();

    // ✅ detect fabric
    $isFabric = $cartItem->is_fabric 
        ?? ($cartItem->product->is_fabric ?? 0);

    // ✅ quantity rule
    if ($isFabric == 1) {
        $quantity = (float) $request->quantity; // meters allowed
    } else {
        $quantity = (int) round($request->quantity); // normal product
    }

    // ✅ update values
    $cartItem->quantity = $quantity;
    $cartItem->subtotal = $cartItem->price * $quantity;
    $cartItem->save();

    return response()->json([
        'success'  => true,
        'message'  => 'Quantity updated',
        'quantity' => $quantity,
        'subtotal' => (float) $cartItem->subtotal
    ]);
}


       public function index()
    {
        $cart = Cart::with([
            'items.product:id,name,master_catalogue_name,master_catalogue_code',
            'items.variant.primaryImage',
            'coupon:id,code,type,value' // ✅ load coupon
        ])->where('user_id', auth()->id())->first();

        if (!$cart || $cart->items->isEmpty()) {
            return response()->json([
                'success' => true,
                'data' => [
                    'items' => [],
                    'coupon' => null,
                    'price_details' => [
                        'sub_total' => 0,
                        'delivery_fee' => 0,
                        'discount' => 0,
                        'total' => 0
                    ]
                ]
            ]);
        }

        $items = $cart->items->map(function ($item) {

            $image = $item->variant->primaryImage->image_path ?? null;
            // ✅ quantity format based on fabric
            $quantity = $item->is_fabric == 1 ? (float) $item->quantity : (int) round($item->quantity);
            return [
                'cart_item_id' => $item->id,
                'is_sample' => $item->is_sample ,
                'is_fabric' => $item->is_fabric ,
                'product_id'   => $item->product_id,
                'name'         => $item->product->name,
                 'catalog'         => $item->product->master_catalogue_name.$item->product->master_catalogue_code,            
                'variant' => [
                    'variant_id' => $item->variant_id,
                      'sku'         => $item->variant->sku,
                    'color'      => $item->variant->color_name ?? null,
                    'image'      => $image ? asset($image) : null
                ],
                'price'    => (float) $item->price,
                'quantity' => $quantity, // ✅ updated here
                'subtotal' => (float) $item->subtotal

            ];
        });

        // ✅ totals
        $subTotal    = $cart->items->sum('subtotal');
        $deliveryFee = 0;
        $discount    = $cart->discount ?? 0;

        $total = max(0, $subTotal + $deliveryFee - $discount);

        return response()->json([
            'success' => true,
            'data' => [
                'items' => $items,

                // ✅ coupon details
                'coupon' => $cart->coupon ? [
                    'coupon_id' => $cart->coupon->id,
                    'code'      => $cart->coupon->code,
                    'type'      => $cart->coupon->type,
                    'value'     => (float) $cart->coupon->value
                ] : null,

                'price_details' => [
                    'sub_total'    => (float) $subTotal,
                    'delivery_fee' => (float) $deliveryFee,
                    'discount'     => (float) $discount,
                    'total'        => (float) $total
                ]
            ]
        ]);
    }



    public function remove($cartItemId)
    {
        $cartItem = CartItem::where('id', $cartItemId)
            ->whereHas(
                'cart',
                fn($q) =>
                $q->where('user_id', auth()->id())
            )->firstOrFail();

        $cartItem->delete();

        return response()->json([
            'success' => true,
            'message' => 'Item removed from cart'
        ]);
    }

    public function applyCoupon(Request $request)
    {
        $request->validate([
            'coupon_code' => 'required|string'
        ]);

        $user = auth()->user();

        $cart = Cart::with(['items.product', 'coupon'])
            ->where('user_id', $user->id)
            ->first();


        if (!$cart || $cart->items->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'Cart is empty'
            ], 400);
        }

        $coupon = Coupon::where('code', $request->coupon_code)
            ->where('is_active', 1)
            ->whereDate('start_date', '<=', now())
            ->where(function ($q) {
                $q->whereNull('end_date')
                    ->orWhereDate('end_date', '>=', now());
            })
            ->first();

        if (!$coupon) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired coupon'
            ], 404);
        }

        // Cart subtotal
        $cartSubtotal = $cart->items->sum(fn($item) => $item->price * $item->quantity);

        // Min purchase check
        if ($coupon->min_purchase && $cartSubtotal < $coupon->min_purchase) {
            return response()->json([
                'success' => false,
                'message' => 'Minimum purchase ₹' . $coupon->min_purchase . ' required'
            ], 422);
        }

        // First-time user
        if ($coupon->first_time_user_only && $user->orders()->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'Coupon only for first-time users'
            ], 422);
        }

        // 🔥 Calculate Discount
        if ($coupon->type === 'percentage') {
            $discount = ($cartSubtotal * $coupon->value) / 100;

            if ($coupon->max_discount_amount) {
                $discount = min($discount, $coupon->max_discount_amount);
            }
        } else {
            $discount = $coupon->value;
        }

        // Prevent over-discount
        $discount = min($discount, $cartSubtotal);

        // Final total
        $finalTotal = $cartSubtotal - $discount;

        // Save coupon to cart
        // overwrite existing coupon automatically (change coupon)
        $cart->coupon_id = $coupon->id;
        $cart->discount  = $discount;
        $cart->total     = $finalTotal;
        $cart->save();

        return response()->json([
            'success' => true,
            'message' => 'Coupon applied successfully',
            'data' => [
                'coupon' => [
                    'code' => $coupon->code,
                    'type' => $coupon->type,
                    'value' => $coupon->value,
                ],
                'price_details' => [
                    'sub_total' => $cartSubtotal,
                    'discount'  => round($discount, 2),
                    'total'     => round($finalTotal, 2),
                ]
            ]
        ]);
    }

    public function removeCoupon()
    {
        $cart = Cart::where('user_id', auth()->id())->firstOrFail();

        $cart->update([
            'coupon_id' => null,
            'discount' => 0,
            'total' => 0
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Coupon removed'
        ]);
    }
}
