<?php

namespace App\Http\Controllers\api\vendor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Zone;
use App\Models\Vendors;
use App\Models\Driver;
use App\Models\Vehicle;
use App\Models\VehicleCategory;
use App\Models\BrandVehicle;
use App\Models\Review;
use App\Models\BookingHistory;
use Illuminate\Database\QueryException;
use Throwable;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\File;
use App\Models\Notifications;

class VendorController extends Controller
{
    public function getVehicles(Request $request)
    {
        try {
            $search = $request->search_item;

            $vehicles = Vehicle::join('vehicle_categories', 'vehicles.vehicle_type', '=', 'vehicle_categories.id')
                ->join('brand_vehicles', 'brand_vehicles.id', '=', 'vehicles.model')
                ->select(
                    'vehicles.id',
                    'vehicles.vehicle_make',
                    'vehicles.year_of_registration',
                    'vehicles.seats',
                    'vehicles.color',
                    'vehicles.fuel_type',
                    'vehicles.is_available',
                    'vehicles.image',
                    'vehicle_categories.name as vehicle_type',
                    'brand_vehicles.model_name as model'
                )
                ->where('vehicles.vendor_id', auth()->user()->id)
                ->where('vehicles.status', 'ACTIVE')
                ->when($search, function ($query) use ($search) {
                    $query->where(function ($q) use ($search) {
                        $q->where('brand_vehicles.model_name', 'like', "%{$search}%")
                            ->orWhere('vehicle_categories.name', 'like', "%{$search}%");
                    });
                })
                ->get();


            $gallery_images = [];
            if (count($vehicles) > 0) {
                foreach ($vehicles as $vehicle) {
                    if ($vehicle->image != NULL) {

                        foreach (json_decode($vehicle->image) as $i => $img) {
                            if ($i == 3) {
                                break; // Stop looping once 3 images collected
                            }
                            $gallery_images[$i] = asset('uploads/vehicle/' . $img); // or Storage::url($img) if using storage                            

                        }
                    }

                    $vehicle->image = $gallery_images;
                }
            }


            return response()->json([
                'status' => true,
                'cars' => $vehicles,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function getVehicleDetails(Request $request)
    {
        try {
            $vehicle = Vehicle::join('vehicle_categories', 'vehicles.vehicle_type', '=', 'vehicle_categories.id')
                ->join('brand_vehicles', 'brand_vehicles.id', '=', 'vehicles.model')
                ->select(
                    'vehicles.*',
                    'vehicle_categories.name as vehicle_type',
                    'brand_vehicles.model_name as model'
                )
                ->where('vehicles.id', $request->vehicle_id)
                ->first();

            $gallery_images = [];

            if ($vehicle->image != NULL) {
                foreach (json_decode($vehicle->image) as $i => $img) {
                    $gallery_images[$i] = asset('uploads/vehicle/' . $img); // or Storage::url($img) if using storage                                                  
                }
            }

            $vehicle->image = $gallery_images;
            $vehicle->makeHidden(['created_at', 'updated_at']);
            return response()->json([
                'status' => true,
                'data' => $vehicle,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }
    public function addvehicle(Request $request)
    {
        try {
            // print_r($request->all());
            // die();
            if (Vehicle::where('plate_number', $request->plate_number)->first() != NULL) {
                return response()->json(['error' => 'Vehicle Plate number already Exist '], 500);
            }
            $image = '';
            $licence_image = '';


            $imagePaths = [];

            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $file) {
                    $filename = Str::random(20) . '.' . $file->getClientOriginalExtension();
                    $destinationPath = public_path('uploads/vehicle');

                    // Create the folder if it doesn't exist
                    if (!file_exists($destinationPath)) {
                        mkdir($destinationPath, 0755, true);
                    }

                    $file->move($destinationPath, $filename);
                    $imagePaths[] =  $filename; // Relative public path
                }
            }

            $res = Vehicle::insert([
                'vendor_id' => auth()->user()->id,
                'driver_id' => $request->driver_id,
                'model' => $request->brand,
                'vehicle_make' => $request->vehicle_make,
                'year_of_registration' => $request->year_of_registration,
                'seats' => $request->seats,
                'color' => $request->color,
                'fuel_type' => $request->fuel_type,
                'plate_number' => $request->plate_number,
                'reg_validity' => date('Y-m-d', strtotime($request->reg_validity)),
                'insurance_validity' => date('Y-m-d', strtotime($request->insurance_validity)),
                'tax_validity' => date('Y-m-d', strtotime($request->tax_validity)),
                'image' => json_encode($imagePaths),
                'vehicle_type' => BrandVehicle::where('id', $request->brand)->pluck('vehicle_type')->first(),

            ]);
            if ($res == 1) {
                return response()->json([
                    'status' => true,
                    'message' => 'Vehicle added successfully',
                ]);
            }
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }


    public function getDrivers(Request $request)
    {
        try {
            $search = $request->search_item;

            $drivers = Driver::select('id', 'name', 'image', 'zone', 'is_availabile_status')
                ->where('vendor_id', auth()->user()->id)
                //    ->where('status', 'ACTIVE')
                ->where('name', 'like', "%{$search}%")
                ->get();


            if (count($drivers) > 0) {
                foreach ($drivers as $driver) {
                    if ($driver->image == NULL) {
                        $driver->image = asset('assets/images/user.jpg');
                    } else {
                        $driver->image = asset('uploads/driver/' . $driver->image);
                    }
                    $driver->zone = Zone::where('id', $driver->zone)->pluck('name')->first();
                    $driver->total_trips = count(BookingHistory::where('driver_id', $driver->id)->get());
                    $driver->review_rate = DB::table('reviews')
                        ->where('driver_id', $driver->id)
                        ->avg('driver_rating');
                }
            }


            return response()->json([
                'status' => true,
                'drivers' => $drivers,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }


    public function adddriver(Request $request)
    {
        try {
            $driver = Driver::where('phone', $request->phone)->first();

            $image = '';
            $licence_image = '';
            if ($request->photo) {
                $rand = rand(100, 999);
                $image = time() . '.' . $request->photo->extension();
                $imageOriginalName = $request->photo->getClientOriginalName();
                $request->photo->move(public_path('uploads/driver/'), $image);
            }
            if ($request->licence) {
                $rand = rand(100, 999);
                $licence_image = time() . '.' . $request->licence->extension();
                $imageOriginalName = $request->licence->getClientOriginalName();
                $request->licence->move(public_path('uploads/driver/licence'), $licence_image);
            }

            if ($driver != null) {
                $res = Driver::where('id', $driver->id)->update([
                    'name' => $request->name,
                    'address' => $request->address,
                    'aadhar_no' => $request->aadhar_no,
                    'image' => $image,
                    'licence_number' => $request->licence_number,
                    'licence_validity' => $request->licence_validity,
                    'licence' => $licence_image,
                    'zone' => $request->zone_id
                ]);
                // return response()->json(['error' => 'Driver with phone number ' . $request->phone . ' already exist.'], 500);
            } else {
                $res = Driver::insert([
                    'vendor_id' => auth()->user()->id,
                    'name' => $request->name,
                    'phone' => $request->phone,
                    'address' => $request->address,
                    'aadhar_no' => $request->aadhar_no,
                    'image' => $image,
                    'licence_number' => $request->licence_number,
                    'licence_validity' => $request->licence_validity,
                    'licence' => $licence_image,
                    'zone' => $request->zone_id
                ]);
            }

            if ($res == 1) {
                return response()->json([
                    'status' => true,
                    'message' => 'Driver added successfully',
                ]);
            }
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function getZones(Request $request)
    {
        try {
            $zones = Zone::select('id', 'name')->where('status', 'ACTIVE')->get();


            return response()->json([
                'status' => true,
                'zones' => $zones,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function getvehiclecategories(Request $request)
    {

        try {
            $VehicleCategory = VehicleCategory::select('id', 'name')->where('status', 'ACTIVE')->get();


            return response()->json([
                'status' => true,
                'VehicleCategory' => $VehicleCategory,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

    public function getDriverDetails(Request $request)
    {
        try {

            $driver = Driver::where('id', $request->driver_id)->first();
            $driver->makeHidden(['created_at', 'updated_at']);

            if ($driver->image == NULL) {
                $driver->image = asset('assets/images/user.jpg');
            } else {
                $driver->image = asset('uploads/driver/' . $driver->image);
            }

            if ($driver->licence) {
                $driver->licence = asset('uploads/driver/licence/' . $driver->licence);
            }
            $driver->zone = Zone::where('id', $driver->zone)->pluck('name')->first();
            $driver->total_rides = count(BookingHistory::where('driver_id', $driver->id)->where('booking_status', '!=', 'Canceled')->get());
            $driver->completed_trips = count(BookingHistory::where('driver_id', $driver->id)->where('booking_status', 'Completed')->get());
            $driver->canceled_trips = count(BookingHistory::where('driver_id', $driver->id)->where('booking_status', 'Canceled')->get());
            $driver->review_rate = DB::table('reviews')
                ->where('driver_id', $driver->id)
                ->avg('driver_rating');



            return response()->json([
                'status' => true,
                'driver' => $driver,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }



    public function notifications(Request $request)
    {
        try {
            $data = Notifications::where('vendor_id', auth()->user()->id)
                ->get();

            return response()->json([
                'status' => true,
                'data' => $data,
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }

      public function deletedriver(Request $request)
    {
        try {
        $data = Driver::where('id', $request->driver_id)->first();
        if ((file_exists(public_path('uploads/driver/' . $data->image))) && ($data->image != NULL)) {
            unlink("uploads/driver/" . $data->image);
        }

        $res = Driver::where('id', $request->driver_id)->delete();
        if ($res == 1){
         return response()->json([
                'status' => true,
                'message' => "Driver deleted successfully",
            ]);
        }
        } catch (QueryException $e) {
            return response()->json(['error' => 'Database error ' . $e], 500);
        } catch (Throwable $e) {
            return response()->json(['error' => 'Something went wrong ' . $e], 500);
        }
    }
}
