<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Event;
use App\Models\Stadium;
use App\Models\Team;
use Illuminate\Support\Facades\File;
use Yajra\DataTables\Facades\DataTables;

class EventController extends Controller
{
    // GET (List for DataTable)
  public function getEvents()
{
    $events = Event::with([
        'stadium:id,name',
        'teamA:id,name,short_name,flag',
        'teamB:id,name,short_name,flag'
    ])->select('*');

    return datatables()->of($events)
        ->addIndexColumn()
        ->addColumn('stadium_name', fn($e) => $e->stadium->name ?? '-')
        ->addColumn('team_a', function($e){
            if ($e->teamA) {
                $flag = $e->teamA->flag ? '<img src="'.asset($e->teamA->flag).'" width="30" class="me-1"> ' : '';
                return $flag . e($e->teamA->name);
            }
            return '-';
        })
        ->addColumn('team_b', function($e){
            if ($e->teamB) {
                $flag = $e->teamB->flag ? '<img src="'.asset($e->teamB->flag).'" width="30" class="me-1"> ' : '';
                return $flag . e($e->teamB->name);
            }
            return '-';
        })
        ->addColumn('banner', fn($e) => $e->banner_image ? '<img src="'.asset($e->banner_image).'" width="60">' : '-')
        ->addColumn('action', function($e){
            return '
                <button class="btn btn-sm btn-primary editEvent" data-id="'.$e->id.'">Edit</button>
                <button class="btn btn-sm btn-danger deleteEvent" data-id="'.$e->id.'">Delete</button>
            ';
        })
        ->rawColumns(['banner', 'team_a', 'team_b', 'action'])
        ->make(true);
}


    // Store
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:cricket,football,concert,other',
            'stadium_id' => 'nullable|exists:stadiums,id',
            'team_a_id' => 'nullable|exists:teams,id',
            'team_b_id' => 'nullable|exists:teams,id|different:team_a_id',
            'start_time' => 'required|date',
            'end_time' => 'nullable|date|after_or_equal:start_time',
            'status' => 'required|string',
            'match_format' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'banner_image' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
        ]);

        $bannerPath = null;
        if ($request->hasFile('banner_image')) {
            $image = $request->banner_image;
            $fileName = time().rand(100,999).'.'.$image->extension();
            $uploadPath = public_path('assets/upload/events/');
            if (!File::exists($uploadPath)) File::makeDirectory($uploadPath,0755,true);
            $image->move($uploadPath, $fileName);
            $bannerPath = 'assets/upload/events/'.$fileName;
        }

        Event::create([
            'name' => $request->name,
            'type' => $request->type,
            'stadium_id' => $request->stadium_id,
            'team_a_id' => $request->team_a_id,
            'team_b_id' => $request->team_b_id,
            'start_time' => $request->start_time,
            'end_time' => $request->end_time,
            'status' => $request->status,
            'match_format' => $request->match_format,
            'banner_image' => $bannerPath,
            'description' => $request->description,
        ]);

        return response()->json(['success' => 'Event added successfully!']);
    }

    // Update
 public function updateEvent(Request $request, $id)
{
    $event = Event::findOrFail($id);

    $request->validate([
        'name'         => 'required|string|max:255',
        'type'         => 'required|in:cricket,football,concert,other',
        'stadium_id'   => 'nullable|exists:stadiums,id',
        'team_a_id'    => 'nullable|exists:teams,id',
        'team_b_id'    => 'nullable|exists:teams,id',
        'start_time'   => 'required|date',
        'end_time'     => 'nullable|date|after_or_equal:start_time',
        'status'       => 'required|string|in:scheduled,live,completed',
        'match_format' => 'nullable|string|max:100',
        'description'  => 'nullable|string',
        'banner_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
    ]);

    $imagePath = $event->banner_image;

    if ($request->hasFile('banner_image')) {
        $image = $request->file('banner_image');
        $imageName = time() . '-' . rand(100, 999) . '.' . $image->extension();
        $uploadPath = public_path('assets/upload/events/');

        if (!File::exists($uploadPath)) {
            File::makeDirectory($uploadPath, 0755, true);
        }

        // Delete old image
        if ($event->banner_image && File::exists(public_path($event->banner_image))) {
            File::delete(public_path($event->banner_image));
        }

        $image->move($uploadPath, $imageName);
        $imagePath = 'assets/upload/events/' . $imageName;
    }

    $event->update([
        'name'         => $request->name,
        'type'         => $request->type,
        'stadium_id'   => $request->stadium_id,
        'team_a_id'    => $request->team_a_id,
        'team_b_id'    => $request->team_b_id,
        'start_time'   => $request->start_time,
        'end_time'     => $request->end_time,
        'status'       => $request->status,
        'match_format' => $request->match_format,
        'description'  => $request->description,
        'banner_image' => $imagePath,
    ]);

    return response()->json(['success' => 'Event updated successfully!']);
}

public function editEvent($id)
{
    $event = Event::with(['stadium', 'teamA', 'teamB'])->findOrFail($id);
    return response()->json(['event' => $event]);
}

    // Delete
    public function destroy($id)
    {
        $event = Event::findOrFail($id);
        if ($event->banner_image && File::exists(public_path($event->banner_image))) {
            File::delete(public_path($event->banner_image));
        }
        $event->delete();
        return response()->json(['success' => 'Event deleted successfully!']);
    }
}
