<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\File;
use App\Models\Category;
use Yajra\DataTables\Facades\DataTables;

class CategoryController extends Controller
{
    public function addcategory(Request $request)
    {
        // Validate inputs
        $request->validate([
            'name'    => 'required|string|max:255',
            'country' => 'required|exists:countries,id',
            'photo'   => 'nullable|image|mimes:jpg,jpeg,png|max:4096',
        ]);

        $slug = Str::slug($request->name);

        $imagePath = null;

        if ($request->hasFile('photo')) {
            $image = $request->photo;
            $imageName = time() . '-' . rand(100,999) . '.' . $image->extension();

            $uploadPath = public_path('assets/upload/category/');
            
            // Create folder if not exists
            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0755, true);
            }

            // Move uploaded file
            $image->move($uploadPath, $imageName);

            // Save relative path to DB
            $imagePath = 'assets/upload/category/' . $imageName;
        }

        // Save category in DB
        Category::create([
            'country_id' => ucfirst($request->country),
            'name'       => ucfirst($request->name),
            'slug'       => $slug,
            'image'      => $imagePath,
        ]);

       return response()->json(['success' => 'Category added successfully!']);
    }

     public function getCategory()
    {
        $users = Category::orderBy('id', 'DESC')->get();

        return DataTables::of($users)
            ->addIndexColumn()
            ->addColumn('name', function ($data) {
                return   $data->name;
            })
            ->addColumn('country', function ($data) {
                return   $data->country->name;
            })
              ->addColumn('image', function ($data) {
               // return $data->image;
               return   '<img src="'.asset($data->image).'" class="avatar-md">';
            })
            ->addColumn('action', function ($row) {
                return '<a href="" class="btn btn-sm btn-purple editcategory" data-id="'.$row->id.'">Edit</a>
                <a href="" class="btn btn-sm btn-danger deletecategory" data-id="' . $row->id . '">Delete</a>';
            })
            ->escapeColumns([])
            ->make(true);
    }

        // Delete category (AJAX)
    public function destroy($id)
    {
        $category = Category::findOrFail($id);
        
    // Delete image from folder if exists
    if ($category->image && File::exists(public_path($category->image))) {
        File::delete(public_path($category->image));
    }

        $category->delete();
        return response()->json(['success' => 'Category deleted successfully.']);
    }

    public function update(Request $request, $id)
{
      $category = Category::findOrFail($id);

    $request->validate([
        'name'    => 'required|string|max:255',
        'country' => 'required|exists:countries,id',
    ]);

    $slug = Str::slug($request->name);
    $imagePath = $category->image; // default: existing image

    // Handle image upload if a new file is provided
    if ($request->hasFile('photo')) {
        $image = $request->photo;
        $imageName = time() . '-' . rand(100, 999) . '.' . $image->extension();
        $uploadPath = public_path('assets/upload/category/');

        // Create folder if it doesn't exist
        if (!File::exists($uploadPath)) {
            File::makeDirectory($uploadPath, 0755, true);
        }

        // Delete old image if exists
        if ($category->image && File::exists(public_path($category->image))) {
            File::delete(public_path($category->image));
        }

        // Move new image
        $image->move($uploadPath, $imageName);
        $imagePath = 'assets/upload/category/' . $imageName;
    }

    // Update category
    $category->update([
        'name'       => ucfirst($request->name),
        'slug'       => $slug,
        'country_id' => $request->country,
        'image'      => $imagePath,
    ]);

    return response()->json(['success' => 'Category updated successfully!']);
}
}
