<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use App\Models\Banner;
use App\Models\Category;
use App\Models\Products;
use Yajra\DataTables\Facades\DataTables;

class BannerController extends Controller
{
    // Add Banner
    public function addBanner(Request $request)
    {
        $request->validate([
            'title'       => 'nullable|string|max:255',
            'category_id' => 'nullable|exists:categories,id',
            'product_id'  => 'nullable|exists:products,id',
            'image'       => 'required|image|mimes:jpg,jpeg,png|max:4096',
        ]);

        $imagePath = null;
        if ($request->hasFile('image')) {
            $image = $request->image;
            $imageName = time() . '-' . rand(100, 999) . '.' . $image->extension();
            $uploadPath = public_path('assets/upload/banner/');
            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0755, true);
            }
            $image->move($uploadPath, $imageName);
            $imagePath = 'assets/upload/banner/' . $imageName;
        }

        Banner::create([
            'title'       => ucfirst($request->title),
            'category_id' => $request->category_id,
            'product_id'  => $request->product_id,
            'image'       => $imagePath,
            'status'      => 1,
        ]);

        return response()->json(['success' => 'Banner added successfully!']);
    }

    // Get Banner list for DataTable
    public function getBanner()
    {
        $banners = Banner::with(['category', 'product'])->orderBy('id', 'DESC')->get();

        return DataTables::of($banners)
            ->addIndexColumn()
            ->addColumn('title', fn($b) => $b->title ?? '-')
            ->addColumn('category', fn($b) => $b->category?->name ?? '-')
            ->addColumn('product', fn($b) => $b->product?->name ?? '-')
            ->addColumn('image', fn($b) => '<img src="' . asset($b->image) . '" class="avatar-md rounded">')
            ->addColumn('status', function ($b) {
                $btnClass = $b->status ? 'btn-success' : 'btn-danger';
                $btnText = $b->status ? 'Active' : 'Inactive';
                return '<button class="btn btn-sm toggleStatusBtn ' . $btnClass . '" title="Click to change status" data-id="' . $b->id . '">' . $btnText . '</button>';
            })
            ->addColumn('action', function ($row) {
                return '
                    <a href="#" class="btn btn-sm btn-purple editbanner" data-id="' . $row->id . '">Edit</a>
                    <a href="#" class="btn btn-sm btn-danger deletebanner" data-id="' . $row->id . '">Delete</a>
                ';
            })
            ->escapeColumns([])
            ->make(true);
    }

    // Delete Banner
    public function destroy($id)
    {
        $banner = Banner::findOrFail($id);

        if ($banner->image && File::exists(public_path($banner->image))) {
            File::delete(public_path($banner->image));
        }

        $banner->delete();

        return response()->json(['success' => 'Banner deleted successfully.']);
    }

    // Update Banner

    public function update(Request $request, $id)
    {
        $banner = Banner::findOrFail($id);

        // Validate input
        $request->validate([
            'title'       => 'nullable|string|max:255',
            'category_id' => 'nullable|exists:categories,id',
            'product_id'  => 'nullable|exists:products,id',
            'image'       => 'nullable|image|mimes:jpg,jpeg,png|max:4096',
        ]);

        // 🔴 If category is empty → return error
        if (!$request->filled('category_id')) {
            return response()->json([
                'error' => 'Please select a category before saving.'
            ], 422);
        }

        // Handle image upload
        $imagePath = $banner->image;
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '-' . rand(100, 999) . '.' . $image->extension();
            $uploadPath = public_path('assets/upload/banner/');

            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0755, true);
            }

            if ($banner->image && File::exists(public_path($banner->image))) {
                File::delete(public_path($banner->image));
            }

            $image->move($uploadPath, $imageName);
            $imagePath = 'assets/upload/banner/' . $imageName;
        }

        // Update values
        $banner->update([
            'title'       => ucfirst($request->title ?? $banner->title),
            'category_id' => $request->category_id,
            'product_id'  => $request->product_id ?? null,
            'image'       => $imagePath,
        ]);

        return response()->json(['success' => 'Banner updated successfully!']);
    }


    public function changeStatus($id)
    {
        $banner = Banner::findOrFail($id);
        $banner->status = !$banner->status; // toggle between 1 and 0
        $banner->save();

        return response()->json([
            'success' => true,
            'status' => $banner->status,
            'message' => 'Banner status updated successfully!',
        ]);
    }
}
