<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Products;
use App\Models\ProductImage;
use App\Models\Category;
use App\Models\SubCategory;
use App\Models\Country;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Yajra\DataTables\Facades\DataTables;

class ProductController extends Controller
{
    public function index()
    {
        return view('admin.product', [
            'countries' => Country::all()
        ]);
    }

    public function getProducts(Request $request)
    {
        if ($request->ajax()) {
            $data = Products::with(['category', 'subCategory', 'images', 'countries'])
            ->orderBy('id','DESC')
            ->get();

            return DataTables::of($data)
                ->addIndexColumn()
                ->addColumn('category', fn($row) => $row->category->name ?? '-')
                ->addColumn('subcategory', fn($row) => $row->subCategory->name ?? '-')
                ->addColumn('country', function ($data) {
                return $data->countries->pluck('name')->implode(', '); // Multiple countries
            })
                ->addColumn('images', function ($row) {
                    if ($row->images && count($row->images)) {
                        $imgTag = '';
                        foreach ($row->images as $img) {
                            $imgTag .= '<img src="' . asset('uploads/products/' . $img->image) . '" width="50" height="50" style="margin:2px;">';
                        }
                        return $imgTag;
                    }
                    return 'No Image';
                })
                ->addColumn('is_trending', fn($row) => $row->is_trending ? '<span class="badge bg-warning">Yes</span>' : 'No')
                ->addColumn('status', fn($row) => $row->status ? '<span class="badge bg-success">Active</span>' : '<span class="badge bg-danger">Inactive</span>')
                // ->addColumn('action', function ($row) {
                //     return '
                //         <button class="btn btn-sm btn-primary editproduct" data-id="' . $row->id . '">Edit</button>
                //         <button class="btn btn-sm btn-danger deleteproduct" data-id="' . $row->id . '">Delete</button>
                //     ';
                // })
                ->addColumn('action', function ($row) {
                     // Trending button
                $trendingBtnClass = $row->is_trending ? 'btn-purple' : 'btn-secondary';
                $trendingBtnText = $row->is_trending ? '<i class="bx bxs-star"></i> Trending' : 'Set Trending';
                $trendingBtn = '<button class="btn btn-sm toggleTrending ' . $trendingBtnClass . '" data-id="' . $row->id . '">' . $trendingBtnText . '</button>';
                    $editBtn = '<a class="btn btn-sm btn-primary " href="'.route('edit-product',['id'=>$row->id]).'"><iconify-icon icon="solar:pen-2-broken" class="align-middle fs-18"></iconify-icon> </a>';
            //    $editBtn = '<button class="btn btn-sm btn-primary editproduct" data-id="' . $row->id . '"><iconify-icon icon="solar:pen-2-broken" class="align-middle fs-18"></iconify-icon> </button>';
                $deleteBtn = '<button class="btn btn-sm btn-danger deleteProduct" data-id="' . $row->id . '"><iconify-icon icon="solar:trash-bin-minimalistic-2-broken" class="align-middle fs-18"></iconify-icon></button>';

               
                return $editBtn . ' ' . $deleteBtn . ' ' . $trendingBtn;
            })
                ->rawColumns(['images', 'is_trending', 'status', 'action'])
                ->make(true);
        }

       // return view('admin.product');
    }

    public function store(Request $request)
{
    // Base validation
    $request->validate([
        'category_id'        => 'required|exists:categories,id',
        'sub_category_id'    => 'nullable|exists:sub_categories,id',
      //  'country_id'         => 'required|exists:countries,id',
       'country'    => 'required|array',           // Accept multiple countries
            'country.*'  => 'exists:countries,id',
        'name'               => 'required|string|max:255',
        'description'        => 'nullable|string',
        'price'              => 'required|numeric|min:0',
        'discount_price'     => 'nullable|numeric|min:0',
        'is_trending'        => 'nullable|boolean',
        'images.*'           => 'image|mimes:jpg,jpeg,png,webp|max:2048',
        'product_type'       => 'required|in:hamper,customized_gift,gift_voucher,standard',

        // Conditional validations
        'custom_photo'       => 'required_if:product_type,customized_gift|image|max:2048',
        'voucher_code' => 'nullable|required_if:product_type,gift_voucher|unique:products,voucher_code',

                // Gift voucher
        'voucher_amount'     => 'nullable|required_if:product_type,gift_voucher|numeric|min:0',
        'voucher_brand'      => 'nullable|required_if:product_type,gift_voucher|string|max:255',
        'voucher_start_date' => 'nullable|required_if:product_type,gift_voucher|date',
        'voucher_end_date'   => 'nullable|required_if:product_type,gift_voucher|date|after_or_equal:voucher_start_date',
        'voucher_value'      => 'nullable|required_if:product_type,gift_voucher|numeric|min:0',
    ]);

    // Generate unique slug
    $slug = Str::slug($request->name);
    $originalSlug = $slug;
    $counter = 1;
    while (Products::where('slug', $slug)->exists()) {
        $slug = $originalSlug . '-' . $counter++;
    }

    $product = new Products();
    $product->category_id         = $request->category_id;
    $product->sub_category_id     = $request->sub_category_id;
    $product->name                = ucfirst($request->name);
    $product->slug                = $slug;
    $product->description         = $request->description;
    $product->price               = $request->price;
    $product->discount_price      = $request->discount_price;
    $product->stock               = $request->stock ?? 0;
    $product->is_trending         = $request->is_trending ?? 0;
    $product->product_type        = $request->product_type;
    $product->status              = 1;

    /**
     * 🔹 Save Fields Based on Product Type
     */
    if ($request->product_type == 'customized_gift') {
        // Upload custom photo
        if ($request->hasFile('custom_photo')) {
            $customPhoto = time() . '-' . rand(100, 999) . '.' . $request->custom_photo->extension();
            $request->custom_photo->move(public_path('uploads/custom'), $customPhoto);
            $product->custom_photo = $customPhoto;
        }
        $product->custom_text     = $request->custom_text;
        $product->custom_message  = $request->custom_message;
        $product->custom_color    = $request->custom_color;
    } 
    elseif ($request->product_type == 'gift_voucher') {
        $product->voucher_amount      = $request->voucher_amount;
        $product->voucher_brand       = $request->voucher_brand;
        $product->voucher_start_date  = $request->voucher_start_date;
        $product->voucher_end_date    = $request->voucher_end_date;
        $product->voucher_value       = $request->voucher_value;
        $product->voucher_code        = $request->voucher_code;
    } 
   

    $product->save();

    /**
     * 🔹 Save Product Images (Multiple)
     */
    if ($request->hasFile('images')) {
        foreach ($request->file('images') as $file) {
            $name = time() . '-' . rand(100, 999) . '.' . $file->extension();
            $file->move(public_path('uploads/products'), $name);
            ProductImage::create([
                'product_id' => $product->id,
                'image'      => $name
            ]);
        }
    }
  $product->countries()->attach($request->country);
    return response()->json(['success' => 'Product added successfully!']);
}


    public function destroy($id)
    {
        $product = Products::findOrFail($id);
        foreach ($product->images as $img) {
            $path = public_path('uploads/products/' . $img->image);
            if (File::exists($path)) File::delete($path);
            $img->delete();
        }
        $product->delete();
        return response()->json(['success' => 'Product deleted successfully.']);
    }

    public function show($id)
    {
        $product = Products::with(['images', 'countries'])->findOrFail($id);
         return view('admin.edit-product',['product'=>$product]);
     //   return response()->json($product);
    }

   public function update(Request $request, $id)
{
    $product = Products::findOrFail($id);

    $request->validate([
        'name'               => 'required|string|max:255',
        'category_id'        => 'required|exists:categories,id',
        'sub_category_id'    => 'nullable|exists:sub_categories,id',
      //  'country_id'         => 'required|exists:countries,id',
         'country'    => 'required|array',
        'country.*'  => 'exists:countries,id',
        'price'              => 'required|numeric|min:0',
        'discount_price'     => 'nullable|numeric|min:0',
        'is_trending'        => 'nullable|boolean',
        'images.*'           => 'image|mimes:jpg,jpeg,png,webp|max:2048',
        'product_type'       => 'required|in:hamper,customized_gift,gift_voucher,standard',

        // Customized gift
        'custom_photo'       => 'nullable|required_if:product_type,customized_gift|image|mimes:jpg,jpeg,png,webp|max:2048',
        'custom_text'        => 'nullable|string',
        'custom_message'     => 'nullable|string',
        'custom_color'       => 'nullable|string',


        // Gift voucher
        'voucher_amount'     => 'nullable|required_if:product_type,gift_voucher|numeric|min:0',
        'voucher_brand'      => 'nullable|required_if:product_type,gift_voucher|string|max:255',
        'voucher_start_date' => 'nullable|required_if:product_type,gift_voucher|date',
        'voucher_end_date'   => 'nullable|required_if:product_type,gift_voucher|date|after_or_equal:voucher_start_date',
        'voucher_value'      => 'nullable|required_if:product_type,gift_voucher|numeric|min:0',
        'voucher_code'       => 'nullable|required_if:product_type,gift_voucher|unique:products,voucher_code,' . $product->id,
    ]);

    $product->update([
        'category_id'        => $request->category_id,
        'sub_category_id'    => $request->sub_category_id,
        'country_id'         => $request->country_id,
        'name'               => ucfirst($request->name),
        'description'        => $request->description,
        'price'              => $request->price,
        'discount_price'     => $request->discount_price,
        'is_trending'        => $request->is_trending ?? 0,
        'product_type'       => $request->product_type,


        // Customized gift
        'custom_text'        => $request->product_type == 'customized_gift' ? $request->custom_text : null,
        'custom_message'     => $request->product_type == 'customized_gift' ? $request->custom_message : null,
        'custom_color'       => $request->product_type == 'customized_gift' ? $request->custom_color : null,

        // Gift voucher
        'voucher_amount'     => $request->product_type == 'gift_voucher' ? $request->voucher_amount : null,
        'voucher_brand'      => $request->product_type == 'gift_voucher' ? $request->voucher_brand : null,
        'voucher_start_date' => $request->product_type == 'gift_voucher' ? $request->voucher_start_date : null,
        'voucher_end_date'   => $request->product_type == 'gift_voucher' ? $request->voucher_end_date : null,
        'voucher_value'      => $request->product_type == 'gift_voucher' ? $request->voucher_value : null,
        'voucher_code'       => $request->product_type == 'gift_voucher' ? $request->voucher_code : null,
        'status' => $request->status,
    ]);

    // Update image for customized gift (single image)
    if ($request->hasFile('custom_photo') && $request->product_type == 'customized_gift') {
        $customPhoto = time() . '-' . rand(100, 999) . '.' . $request->custom_photo->extension();
        $request->custom_photo->move(public_path('uploads/products/custom'), $customPhoto);
        $product->update(['custom_photo' => $customPhoto]);
    }

    // Multiple product images
    if ($request->hasFile('images')) {
        foreach ($request->file('images') as $file) {
            $name = time() . '-' . rand(100, 999) . '.' . $file->extension();
            $file->move(public_path('uploads/products'), $name);
            ProductImage::create([
                'product_id' => $product->id,
                'image'      => $name
            ]);
        }
    }
  // Sync multiple selected countries
        $product->countries()->sync($request->country);
    return response()->json(['success' => 'Product updated successfully!']);
}


    public function updateTrending($id)
{
    $product = Products::findOrFail($id);

    // Toggle trending status (1 → 0 or 0 → 1)
    $product->is_trending = !$product->is_trending;
    $product->save();

    return response()->json([
        'success' => true,
        'message' => $product->is_trending
            ? 'Product marked as trending.'
            : 'Product unmarked as trending.'
    ]);
}


public function getProductsByCategory($category_id)
{
    $products = Products::where('category_id', $category_id)
        ->select('id', 'name')
        ->get();

    return response()->json($products);
}

}
