<?php

namespace App\Http\Controllers\api\vendor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use App\Models\Fare;
use App\Models\Zone;
use App\Models\Driver;
use App\Models\VehicleCategory;
use App\Models\BrandVehicle;
use App\Models\Vehicle;
use App\Models\Settings;
use App\Models\Review;
use App\Models\BookingHistory;
use Illuminate\Database\QueryException;
use Throwable;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Models\Notifications;

class BookingController extends Controller
{
  public function getBookings(Request $request)
  {
    try {

      $bookings = BookingHistory::select('id', 'booking_id', 'departure_date', 'return_date', 'departure_time', 'return_time', 'pick_up', 'drop_off', 'stay_locations', 'booking_status', 'total_fare', 'paid_amount')
        ->where('vendor_id', auth()->user()->id);
      //->get();

      if ($request->filled('status')) {
        $bookings = $bookings->where('booking_status', $request->status);
      }
      $bookings = $bookings->get();
      foreach ($bookings as $item) {
        $raw = $item->stay_locations; // The double-encoded JSON string

        $firstDecode = json_decode($raw, true);              // returns a JSON string
        $locationsArray = json_decode($firstDecode, true);   // returns the actual PHP array

        $locationString = is_array($locationsArray) ? implode(', ', $locationsArray) : '';

        $item->stay_locations = $locationString;
        //  $item->stay_locations_count = is_array($locationsArray) ? count($locationsArray) : 0;

        $departure = Carbon::parse($item->departure_date);
        $return = Carbon::parse($item->return_date);

        // Total days (inclusive)
        $totalDays = $departure->diffInDays($return) + 1; // +1 to include both start and end dates
        $nightCount = $totalDays - 1;

        // Final title
        $tripTitle = "{$nightCount}NIGHTS {$totalDays}DAYS";
        $item->title = $tripTitle;
      }

      return response()->json([
        'status' => true,
        'bookings' => $bookings,
      ]);
    } catch (QueryException $e) {
      return response()->json(['error' => 'Database error ' . $e], 500);
    } catch (Throwable $e) {
      return response()->json(['error' => 'Something went wrong ' . $e], 500);
    }
  }

  public function getBookingDetails(Request $request)
  {
    try {

      // $booking= BookingHistory::select('id','vehicle_id','driver_id','booking_id','total_km','departure_date','return_date','departure_time','return_time','pick_up','drop_off','stay_date','stay_locations','adults','children','client_name','client_mobile_number','client_email','booking_status','total_fare','paid_amount')
      // ->where('id',$request->booking_id)
      // ->first();
      $booking =  BookingHistory::select(
        'booking_histories.*',
        'reviews.driver_rating',
        'reviews.car_rating',
        'reviews.comment'
      )
        ->leftJoin('reviews', 'booking_histories.id', '=', 'reviews.booking_id')
        ->where('booking_histories.id', $request->booking_id)
         ->where('vendor_id', auth()->user()->id)
        ->first();
      
        if ($booking == null) {
        return response()->json([
          'status' => true,
          'data' => $booking,
        ]);
      }
      $booking->makeHidden(['created_at', 'updated_at']);
      // ********** stay location  *********
      $raw = $booking->stay_locations; // The double-encoded JSON string
      $firstDecode = json_decode($raw, true);              // returns a JSON string
      $locationsArray = json_decode($firstDecode, true);   // returns the actual PHP array
      $booking->stay_locations = $locationsArray;
      // ********** stay date end  *********

      // ********** find titile  *********
      $departure = Carbon::parse($booking->departure_date);
      $return = Carbon::parse($booking->return_date);

      // Total days (inclusive)
      $totalDays = $departure->diffInDays($return) + 1; // +1 to include both start and end dates
      $nightCount = $totalDays - 1;

      // Final title
      $tripTitle = "{$nightCount}NIGHTS {$totalDays}DAYS";
      $booking->title = $tripTitle;
      // ********** find titile end  *********

      // ********** stay date  *********
      $raw = $booking->stay_date; // The double-encoded JSON string

      $firstDecode = json_decode($raw, true);              // returns a JSON string
      $locationsArray = json_decode($firstDecode, true);   // returns the actual PHP array

      $booking->stay_date = $locationsArray;
      // ********** stay date end *********

      $vehicle = Vehicle::join('drivers', 'vehicles.driver_id', '=', 'drivers.id')
        ->join('vendors', 'vehicles.vendor_id', '=', 'vendors.id')
        ->leftJoin('reviews', 'vehicles.id', '=', 'reviews.vehicle_id')
        ->leftJoin('booking_histories', 'vehicles.id', '=', 'booking_histories.vehicle_id')
        ->select(
          'vehicles.id',
          'vehicles.seats',
          'vehicles.fuel_type',
          'vehicles.model',
          'vehicles.vehicle_make',
          'vehicles.image',
          'vendors.name as vendor_name',
          'vendors.vendor_image',
          DB::raw('AVG(reviews.overall_rating) as avg_rating'),
          DB::raw('COUNT(booking_histories.id) as total_trips')
        )
        ->where('vehicles.id', $booking->vehicle_id)
        ->groupBy(
          'vehicles.id',
          'vehicles.seats',
          'vehicles.fuel_type',
          'vehicles.model',
          'vehicles.vehicle_make',
          'vehicles.image',
          'vendors.name',
          'vendors.vendor_image'
        )
        ->first();
  
  if($vehicle!=null){
      $vehicle->model = BrandVehicle::where('id', $vehicle->model)->pluck('model_name')->first();
      foreach (json_decode($vehicle->image) as $i => $img) {
        if ($i == 3) {
          break; // Stop looping once 3 images collected
        }
        $gallery_images[$i] = asset('uploads/vehicle/' . $img); // or Storage::url($img) if using storage
      }

      if ($vehicle->vendor_image == NULL) {
        $vehicle->vendor_image = asset('assets/images/user.jpg');
      } else {
        $vehicle->vendor_image = asset('uploads/vendor/' . $vehicle->vendor_image);
      }
      $vehicle->image = $gallery_images;
      $vehicle->review_rate = 5;


      $booking->cab_agency_details = $vehicle;
  }else{
      $booking->cab_agency_details =[];
  }
      return response()->json([
        'status' => true,
        'booking_details' => $booking,
      ]);
    } catch (QueryException $e) {
      return response()->json(['error' => 'Database error ' . $e], 500);
    } catch (Throwable $e) {
      return response()->json(['error' => 'Something went wrong ' . $e], 500);
    }
  }

  public function assignDriver(Request $request)
  {
    try {
          
      $res = BookingHistory::where('id', $request->booking_id)->update(['driver_id' => $request->driver_id, 'booking_status' => 'Confirmed']);
      if ($res == 1) {

       $booking = BookingHistory::where('id', $request->booking_id)
          ->where('driver_id', $request->driver_id)
          ->first();

        if ($booking) {
        
          $driver = Driver::find($booking->driver_id);
          $vehicle = Vehicle::find($booking->vehicle_id);
          $brandModel = $vehicle ? BrandVehicle::find($vehicle->model) : null;

          $bookingId = $booking->booking_id;
          $driverName = $driver?->name ?? 'N/A';
          $driverContact = $driver?->phone ?? 'N/A';
          $vehicleName = $vehicle?->vehicle_make . ' ' . ($brandModel?->model_name ?? '');
        }
        $curl = curl_init();

        curl_setopt_array($curl, array(
          CURLOPT_URL => 'https://api.connectpanels.com/whatsapp-api/v1.0/customer/120180/bot/0d094a19d57e4eaa/template',
          CURLOPT_RETURNTRANSFER => true,
          CURLOPT_ENCODING => '',
          CURLOPT_MAXREDIRS => 10,
          CURLOPT_TIMEOUT => 0,
          CURLOPT_FOLLOWLOCATION => true,
          CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
          CURLOPT_CUSTOMREQUEST => 'POST',
          CURLOPT_POSTFIELDS => '{"payload":{"name":"booking_confirmation","components":[{"type":"body","parameters":[{"type":"text","text":"' . $bookingId . '"},{"type":"text","text":"' . $driverName . '"},{"type":"text","text":"' . $driverContact . '"},{"type":"text","text":"' . $vehicleName . '"}]}],"language":{"code":"en_US","policy":"deterministic"},"namespace":"66cbcb46_a232_4519_a6e7_b69fd50991bf"},"phoneNumber":"'.$booking->client_mobile_number.'"}',
          CURLOPT_HTTPHEADER => array(
            'Authorization: Basic c4d8e3eb-4e7c-4f9d-a05c-5ff0bf583d29-Ic1uEZu',
            'Content-Type: application/json'
          ),
        ));

        $response = curl_exec($curl);

        curl_close($curl);
        // echo $response;

         Notifications::insert([
                        'booking_id' => $bookingId,
                        'vendor_id' => $vehicle->vendor_id,
                        'driver_id' => $vehicle->driver_id,
                        'booked_user_id' => auth()->user()->id,
                       'notification_title'=> 'Booking Confirmation',
                        'notification' => "Booking Confirmed for Booking ID: $bookingId , Vehicle $vehicleName ", 
                    ]);

        return response()->json([
          'status' => true,
          'message' => "Driver Assigned Successfully",
        ]);
      }
    } catch (QueryException $e) {
      return response()->json(['error' => 'Database error ' . $e], 500);
    } catch (Throwable $e) {
      return response()->json(['error' => 'Something went wrong ' . $e], 500);
    }
  }



  public function getAvailableDrivers(Request $request)
  {
    try {

      $request->validate([
        'departure_date' => 'required|date',
        'return_date' => 'required|date|after_or_equal:departure_date',
      ]);

      $departureDate = date("Y-m-d", strtotime($request->departure_date));
      $returnDate = date("Y-m-d", strtotime($request->return_date));
      $origin = $request->pick_up_location;
      // Step 1: Get coordinates of origin
      $originCoords = $this->getCoordinatesFromAddress($origin);

      // Step 2: Find the zone for origin
      $zones = Zone::all();
      $originZone = null;
      foreach ($zones as $zone) {
        $polygon = json_decode($zone->coordinates, true); // ← fix here
        if ($this->isPointInPolygon($originCoords, $polygon)) {
          $originZone = $zone;
          break;
        }
      }
      // echo $originZone->id;
      $availableDrivers = DB::table('drivers as d')
        ->where('d.vendor_id', auth()->user()->id)
        ->where('d.status', 'ACTIVE')
        //  ->where('d.is_availabile_status', 'False')
        ->where('d.is_availabile_status', 'True')
        ->where('d.zone', $originZone->id)
        ->whereNotExists(function ($query) use ($departureDate, $returnDate) {
          $query->select(DB::raw(1))
            ->from('booking_histories as bh')
            ->whereRaw('bh.driver_id = d.id')
            ->where(function ($q) use ($departureDate, $returnDate) {
              $q->where('bh.departure_date', '<=', $returnDate)
                ->where('bh.return_date', '>=', $departureDate);
            });
        })
        ->select('d.id', 'd.name', 'd.phone', 'd.image')
        //   ->select('d.id', 'd.name')
        ->get();

      foreach ($availableDrivers as $driver) {
        if ($driver->image == NULL) {
          $driver->image = asset('assets/images/user.jpg');
        } else {
          $driver->image = asset('uploads/driver/' . $driver->image);
        }
      }
      return response()->json([
        'success' => true,
        'data' => $availableDrivers
      ]);
    } catch (QueryException $e) {
      return response()->json(['error' => 'Database error ' . $e], 500);
    } catch (Throwable $e) {
      return response()->json(['error' => 'Something went wrong ' . $e], 500);
    }
  }


  public function isPointInPolygon($point, $polygon)
  {
    try {
      $lat = $point['lat'];
      $lng = $point['lng'];
      $inside = false;
      $j = count($polygon) - 1;

      for ($i = 0; $i < count($polygon); $i++) {
        $xi = $polygon[$i]['lat'];
        $yi = $polygon[$i]['lng'];
        $xj = $polygon[$j]['lat'];
        $yj = $polygon[$j]['lng'];

        $intersect = (($yi > $lng) != ($yj > $lng))
          && ($lat < ($xj - $xi) * ($lng - $yi) / ($yj - $yi) + $xi);
        if ($intersect) $inside = !$inside;
        $j = $i;
      }

      return $inside;
    } catch (QueryException $e) {
      return response()->json(['error' => 'Database error ' . $e], 500);
    } catch (Throwable $e) {
      return response()->json(['error' => 'Something went wrong ' . $e], 500);
    }
  }

  public function getCoordinatesFromAddress($address)
  {
    try {
      $apiKey = Settings::pluck('google_map_api_key')->first();
      $response = Http::get("https://maps.googleapis.com/maps/api/geocode/json", [
        'address' => $address,
        'key' => $apiKey,
      ]);

      $data = $response->json();
      if ($data['status'] === 'OK') {
        $location = $data['results'][0]['geometry']['location'];
        return ['lat' => $location['lat'], 'lng' => $location['lng']];
      }

      throw new \Exception("Invalid address or failed to fetch coordinates.");
    } catch (QueryException $e) {
      return response()->json(['error' => 'Database error ' . $e], 500);
    } catch (Throwable $e) {
      return response()->json(['error' => 'Something went wrong ' . $e], 500);
    }
  }
}
