<?php 
namespace App\Http\Controllers;
use App\Models\Location;
use App\Models\Stock;
use Yajra\DataTables\DataTables;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
class StockController extends Controller
{
    public function index()
    {
        $totalItems = Stock::sum('available_qty');
        $inStock = Stock::where('available_qty', '>', 0)->count();
        $outOfStock = Stock::where('available_qty', 0)->count();
        $lowStock = Stock::whereColumn('available_qty', '<=', 'reorder_level')->count();

        return view('admin.stock.index', compact(
            'totalItems',
            'inStock',
            'outOfStock',
            'lowStock'
        ));
    }

    public function getStock(Request $request)
    {
        $locations = Location::withCount([
        'stocks as total_items',
        'stocks as in_stock' => function ($q) {
            $q->where('available_qty', '>', 0);
        },
        'stocks as out_of_stock' => function ($q) {
            $q->where('available_qty', '<=', 0);
        }
    ])->withSum('stocks as total_stock_qty', 'available_qty');

    if ($request->type) {
        $locations->where('type', $request->type);
    }

    return DataTables::of($locations)
        ->addIndexColumn()

        ->addColumn('location', function ($row) {
            return '<strong>'.$row->name.'</strong>';
        })

        ->addColumn('type', function ($row) {
            return '<span class="text-muted">'.$row->type.'</span>';
        })

        ->addColumn('total_items', function ($row) {
            return number_format($row->total_items);
        })

        ->addColumn('in_stock', function ($row) {
            return '<span class="badge bg-success-subtle text-success">Available</span>';
        })

        ->addColumn('out_of_stock', function ($row) {
            return '<span class="badge bg-danger-subtle text-danger">'.$row->out_of_stock.'</span>';
        })

        ->addColumn('total_stock_qty', function ($row) {
            return number_format($row->total_stock_qty ?? 0);
        })

        ->addColumn('action', function ($row) {
            return '
                <a href="'.route('stock.view', $row->id).'" class="btn btn-sm btn-light">
                    <i class="ti ti-eye"></i>
                </a>
                <a href="'.route('stock.edit', $row->id).'" class="btn btn-sm btn-warning ms-1">
                    <i class="ti ti-edit"></i>
                </a>
            ';
        })

        ->rawColumns(['type','location','in_stock','out_of_stock','action'])
        ->make(true);
    }

     public function viewStocks(Location $location, Request $request)
    {
        if ($request->ajax()) {

            $stocks = Stock::with(['productVariant.product'])
                ->where('location_id', $location->id);

            return DataTables::of($stocks)
                ->addIndexColumn()

                ->addColumn('product', function ($row) {
                    return optional($row->productVariant->product)->name ?? '-';
                })

                ->addColumn('variant', function ($row) {
    $variant = $row->productVariant;

    if (!$variant) {
        return '-';
    }

    // Build a nice HTML card for the variant
    $html = '<div class="variant-card p-2 border rounded bg-light">';
    $html .= '<strong>SKU:</strong> ' . $variant->sku . '<br>';
    $html .= '<strong>Batch No:</strong> ' . $variant->batch_no . '<br>';
    $html .= '<strong>MRP:</strong> ₹' . number_format($variant->mrp, 2) . '<br>';
    $html .= '<strong>Offer Price:</strong> <span class="text-success">₹' . number_format($variant->offer_price, 2) . '</span><br>';
    $html .= '<strong>Unit:</strong> ' . $variant->unit . '<br>';
    $html .= '<strong>Dimensions:</strong> ' . $variant->dimensions . '<br>';
    $html .= '<strong>Color:</strong> <span style="display:inline-block;width:15px;height:15px;background:' . $variant->color_code . ';border:1px solid #ccc;margin-right:5px;"></span>' . $variant->color_name . '<br>';
    $html .= '<strong>Material:</strong> ' . $variant->material_or_fabric_composition . '<br>';
    $html .= '<strong>GSM:</strong> ' . $variant->gsm . '<br>';
    $html .= '<strong>Weight:</strong> ' . $variant->weight . ' kg<br>';
    $html .= '<strong>Pattern:</strong> ' . $variant->pattern . '<br>';
    $html .= '<strong>MOQ:</strong> ' . $variant->moq . '<br>';
    $html .= '</div>';

    return $html;
})



                ->addColumn('available_qty', function ($row) {
                    return $row->available_qty;
                })

                ->addColumn('reorder_level', function ($row) {
                    return $row->reorder_level ?? '-';
                })

                ->addColumn('status', function ($row) {
                    if ($row->available_qty <= 0) {
                        return '<span class="badge bg-danger-subtle text-danger">Out of Stock</span>';
                    }

                    if ($row->reorder_level && $row->available_qty <= $row->reorder_level) {
                        return '<span class="badge bg-warning-subtle text-warning">Low Stock</span>';
                    }

                    return '<span class="badge bg-success-subtle text-success">Available</span>';
                })

                ->rawColumns(['status','variant'])
                ->make(true);
        }

        return view('admin.stock.view', compact('location'));
    }

    public function view($id, Request $request)
{
    $location = Location::findOrFail($id);

    if ($request->ajax()) {
        $stocks = Stock::with(['productVariant.product'])
            ->where('location_id', $id);

        return DataTables::of($stocks)
            ->addIndexColumn()
            ->make(true);
    }

    return view('admin.stock.view', compact('location'));
}

 public function edit($id, Request $request)
{
    return view('admin.stock.edit');
}

//************location  ******************/

    public function addlocations(Request $request)
    {
        $request->validate([
            'type' => 'required',
            'name'        => 'required|string|max:255',
        ]);

       
        Location::create([
            'type' => $request->type,
            'name'        => ucfirst($request->name)
        ]);

        return response()->json(['success' => 'Sub location added successfully!']);
    }

    public function getlocations()
    {
        $subcats = Location::orderBy('id', 'DESC')->get();

        return DataTables::of($subcats)
            ->addIndexColumn()
            ->addColumn('name', fn($data) => $data->name)
            ->addColumn('type', fn($data) => $data->type)
          
            ->addColumn('action', function ($row) {
                $image = $row->image ? asset($row->image) : asset('images/no-image.png');

                return '
        <button class="btn btn-sm btn-primary editlocation"
            data-id="' . $row->id . '"
            data-name="' . $row->name . '"
            data-location="' . $row->type . '">
            Edit
        </button>
        <button class="btn btn-sm btn-danger deletelocation"
            data-id="' . $row->id . '">
            Delete
        </button>';
            })

            ->escapeColumns([])
            ->make(true);
    }

    public function update(Request $request, $id)
    {
        $location = Location::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required',
        ]);

    

        $location->update([
            'name' => ucfirst($request->name),
            'type' => $request->type,
        ]);

        return response()->json(['success' => 'Sub location updated successfully!']);
    }

    public function destroy($id)
    {
        $subcat = Location::findOrFail($id);
        $subcat->delete();
        return response()->json(['success' => 'Sub location deleted successfully.']);
    }
}
