<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Coupon;
use App\Models\Category;
use App\Models\Product;
use Yajra\DataTables\Facades\DataTables;

class CouponController extends Controller
{
    // List view
    public function index()
    {
        return view('admin.coupons');
    }

    // Add Coupon
    public function store(Request $request)
    {
        $request->validate([
            'code' => 'required|string|max:50|unique:coupons,code',
            'type' => 'required|in:percentage,fixed',
            'value' => 'required|numeric|min:0',
            'category_id' => 'nullable|exists:categories,id',
            'product_id' => 'nullable|exists:products,id',
            'min_purchase' => 'nullable|numeric|min:0',
            'max_uses' => 'nullable|integer|min:1',
            'per_user_limit' => 'nullable|integer|min:1',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'description' => 'nullable|string|max:255',
        ]);

        Coupon::create($request->all());

        return response()->json(['success' => 'Coupon added successfully!']);
    }

    // Get Coupons (DataTables)
    public function getCoupons()
    {
        $coupons = Coupon::with([ 'category', 'product'])->orderBy('id', 'DESC');

        return DataTables::of($coupons)
            ->addIndexColumn()
            ->addColumn('code', fn($c) => strtoupper($c->code))
            ->addColumn('type_value', fn($c) => $c->type == 'percentage' ? $c->value . '%' : '₹' . $c->value)
            ->addColumn('category', fn($c) => $c->category?->name ?? '-')
            ->addColumn('product', fn($c) => $c->product?->name ?? '-')
            ->addColumn('date_range', fn($c) => ($c->start_date ?? '-') . ' to ' . ($c->end_date ?? '-'))
            ->addColumn('status', function ($c) {
                $btnClass = $c->is_active ? 'btn-success' : 'btn-danger';
                $btnText = $c->is_active ? 'Active' : 'Inactive';
                return '<button class="btn btn-sm toggleStatusBtn ' . $btnClass . '" data-id="' . $c->id . '">' . $btnText . '</button>';
            })
            ->addColumn('action', function ($row) {
                return '
                    <a href="#" class="btn btn-sm btn-purple editCouponBtn" data-id="' . $row->id . '">Edit</a>
                    <a href="#" class="btn btn-sm btn-danger deleteCoupon" data-id="' . $row->id . '">Delete</a>
                ';
            })
            ->escapeColumns([])
            ->make(true);
    }

    // Delete Coupon
    public function destroy($id)
    {
        $coupon = Coupon::findOrFail($id);
        $coupon->delete();
        return response()->json(['success' => 'Coupon deleted successfully.']);
    }

  
    // Update coupon
    public function update(Request $request, $id)
    {
        $coupon = Coupon::findOrFail($id);

        $request->validate([
            'code' => 'required|string|max:50|unique:coupons,code,' . $coupon->id,
            'type' => 'required|in:percentage,fixed',
            'value' => 'required|numeric|min:0',
            'category_id' => 'nullable|exists:categories,id',
            'product_id' => 'nullable|exists:products,id',
            'min_purchase' => 'nullable|numeric|min:0',
            'max_uses' => 'nullable|integer|min:1',
            'per_user_limit' => 'nullable|integer|min:1',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'is_active' => 'boolean',
            'description' => 'nullable|string|max:255',
        ]);

        $coupon->update($request->all());

        return response()->json(['success' => true, 'message' => 'Coupon updated successfully!']);
    }

    // Change active status
    public function changeStatus($id)
    {
        $coupon = Coupon::findOrFail($id);
        $coupon->is_active = !$coupon->is_active;
        $coupon->save();

        return response()->json([
            'success' => true,
            'status' => $coupon->is_active,
        ]);
    }



  

    public function getProducts($category_id)
    {
        $products = Product::where('category_id', $category_id)
            ->where('is_active', 1)
            ->select('id', 'name')
            ->get();

        return response()->json($products);
    }

    
public function edit($id)
{
    $coupon = Coupon::findOrFail($id);
    return response()->json(['data' => $coupon]);
}



}
