<?php


namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\ProductVariant;
use App\Models\ProductImages;
use App\Models\Stock;
use Illuminate\Support\Str;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class ProductController extends Controller
{
  public function storeStepOne(Request $request)
{
    $validated = $request->validate([
        'category_id'     => 'required|exists:categories,id',
        'sub_category_id' => 'required|exists:sub_categories,id',
        'name'            => 'required|string|max:855',
        'description'     => 'nullable|string',

        'length'          => 'nullable|numeric',
        'width'           => 'nullable|numeric',
        'height'          => 'nullable|numeric',

        'features'        => 'nullable|string|max:855',
        'maintenance'     => 'nullable|string|max:855',
        'warranty_certification' => 'nullable|string|max:855',

        'rooms'           => 'nullable|array',
        'rooms.*'         => 'exists:rooms,id',
        'tags'            => 'nullable|array',
        'tags.*'          => 'exists:tags,id',

        'master_catalogue_name' => 'required|string|max:255',
        'master_catalogue_code' => 'required|string|max:100',

        // Extra fields
        'visibility'          => 'required|in:B2C,B2B,Franchise,All',
        'gst_percent'         => 'nullable|numeric|min:0|max:100',
        'gst_inclusive'       => 'required|boolean',
        'is_featured'         => 'required|boolean',
        'related_products'    => 'nullable|array',
        'related_products.*'  => 'exists:products,id',
        'cross_sell_products' => 'nullable|array',
        'cross_sell_products.*'=> 'exists:products,id',
        'upsell_products'     => 'nullable|array',
        'upsell_products.*'   => 'exists:products,id',
        'seo_title'           => 'nullable|string|max:255',
        'seo_description'     => 'nullable|string',
        'search_keywords'     => 'nullable|string',
        'care_instructions'   => 'nullable|string',
        'fast_moving'         => 'nullable|boolean',
        'slow_moving'         => 'nullable|boolean',
    ]);

    // Calculate measurement
    $measurement = null;
    if ($request->length || $request->width || $request->height) {
        $measurement = trim(
            ($request->length ? $request->length . ' x ' : '') .
            ($request->width ? $request->width . ' x ' : '') .
            ($request->height ? $request->height : ''),
            ' x '
        );
    }

    DB::beginTransaction();
    try {
        // Map sub_category_id to subcategory_id in DB
        $productData = $validated;
        $productData['subcategory_id'] = $validated['sub_category_id'];
        unset($productData['sub_category_id']);

        // Add calculated measurement
        $productData['measurement'] = $measurement;

        // Create product
        $product = Product::create($productData);

        // Attach rooms to product_rooms table
        if ($request->filled('rooms')) {
            $product->rooms()->sync($request->rooms);
        }

        // Attach tags to product_tags table
        if ($request->filled('tags')) {
            $product->tags()->sync($request->tags);
        }

        DB::commit();

        return response()->json([
            'success' => true,
            'product_id' => $product->id,
            'message' => 'Step 1 saved successfully.'
        ]);
    } catch (\Exception $e) {
        DB::rollBack();
        return response()->json([
            'success' => false,
            'message' => 'Something went wrong. '.$e,
            'error' => $e->getMessage()
        ], 500);
    }
}


    public function storeStepTwo(Request $request)
    {

        $validated = $request->validate([
            'product_id' => ['required', 'exists:products,id'],
            'sku' => [
                'required',
                Rule::unique('product_variants')
                    ->where(function ($query) use ($request) {
                        return $query->where('product_id', $request->product_id);
                    }),
            ],
            'mrp' => 'required|numeric',
            'offer_price' => 'required|numeric',
            'unit' => 'required|string',
        ]);

        $variant = ProductVariant::create([
            'product_id' => $request->product_id,
            'sku' => $request->sku,
            'batch_no' => $request->batch_no,
            'mrp' => $request->mrp,
            'offer_price' => $request->offer_price,
            'unit' => $request->unit,
            'dimensions' => $request->dimensions,
            'color_name' => $request->color_name,
            'color_code' => $request->color_code,
            'material_or_fabric_composition' => $request->material_or_fabric_composition,
            'gsm' => $request->gsm,
            'weight' => $request->weight,
            'rub_count' => $request->rub_count,
            'moq' => $request->moq,
            'pattern' => $request->pattern,
        ]);

        return response()->json([
            'success' => true,
            'product_variant_id' => $variant->id,
            'message' => 'Variant saved successfully'
        ]);
    }


    public function storeImages(Request $request)
    {
        $request->validate([
            'product_variant_id' => 'required|exists:product_variants,id',
            'images.*' => 'required|image|max:2048'
        ]);

        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $file) {

                $name = time() . '-' . rand(100, 999) . '.' . $file->extension();

                // Destination path
                $destination = public_path('assets/upload/products');

                // Move file
                $file->move($destination, $name);

                ProductImages::create([
                    'product_variant_id' => $request->product_variant_id,
                    // store FULL RELATIVE PATH
                    'image_path' => 'assets/upload/products/' . $name,
                    'type' => 'front'
                ]);
            }
        }


        // foreach ($request->file('images') as $image) {
        //     $path = $image->store('products', 'public');

        //     ProductImages::create([
        //         'product_variant_id' => $request->product_variant_id,
        //         'image_path' => $path,
        //         'type' => 'front'
        //     ]);
        // }

        return response()->json([
            'success' => true,
            'message' => 'Images uploaded successfully'
        ]);
    }

public function storeStock(Request $request)
{
    $validated = $request->validate([
        'product_variant_id' => 'required|exists:product_variants,id',
        'stocks' => 'required|array|min:1',
        'stocks.*.location_id'   => 'required|exists:locations,id',
        'stocks.*.available_qty' => 'required|numeric|min:0',
        'stocks.*.reorder_level' => 'nullable|numeric|min:0',
        'stocks.*.opening_qty'   => 'nullable|numeric|min:0',
        'stocks.*.received_qty'  => 'nullable|numeric|min:0',
        'stocks.*.damaged_qty'   => 'nullable|numeric|min:0',
        'stocks.*.returned_qty'  => 'nullable|numeric|min:0',
    ]);

    foreach ($request->stocks as $stock) {
        Stock::updateOrCreate(
            [
                'product_variant_id' => $request->product_variant_id,
                'location_id'        => $stock['location_id'],
            ],
            [
                'available_qty' => $stock['available_qty'],
                'reorder_level' => $stock['reorder_level'] ?? null,
                'opening_qty'   => $stock['opening_qty'] ?? null,
                'received_qty'  => $stock['received_qty'] ?? null,
                'damaged_qty'   => $stock['damaged_qty'] ?? null,
                'returned_qty'  => $stock['returned_qty'] ?? null,
                'is_available'  => $stock['available_qty'] > 0,
            ]
        );
    }

    return response()->json([
        'success' => true,
        'message' => 'Stock saved successfully'
    ]);
}


    public function index()
    {
        return view('admin.product.product');
    }



    public function getProducts(Request $request)
    {
        $products = Product::with([
            'category',
            'subCategory',
            'variants.images',
            'variants.stocks'
        ])->latest();

        return DataTables::of($products)
            ->addIndexColumn()

            ->addColumn('category', fn($p) => $p->category->name ?? '-')

            ->addColumn('subcategory', fn($p) => $p->subCategory->name ?? '-')
            ->addColumn('images', function ($p) {

                $images = $p->variants->flatMap(fn($v) => $v->images)->take(3);

                if ($images->isEmpty()) {
                    return '-';
                }

                $html = '<div class="d-flex align-items-center gap-1 flex-nowrap">';

                foreach ($images as $i) {

                    $path = is_array($i->image_path)
                        ? implode('/', $i->image_path)
                        : $i->image_path;

                    $url = asset($path);

                    $html .= '<img src="' . $url . '" height="40"
                    class="rounded"
                    style="width:40px;object-fit:cover;">';
                }

                $html .= '</div>';

                return $html;
            })




            ->addColumn('price', function ($p) {
                $variant = $p->variants->first();
                return $variant
                    ? '₹' . $variant->offer_price
                    : '-';
            })

            ->addColumn('stock', function ($p) {
                return $p->variants->sum(
                    fn($v) =>
                    $v->stocks->sum('available_qty')
                );
            })

            ->addColumn('status', function ($p) {
                return $p->is_active
                    ? '<span class="badge bg-success">Active</span>'
                    : '<span class="badge bg-danger">Inactive</span>';
            })

            ->addColumn('action', function ($p) {
                return '
                <a href="' . route('edit-product', ['id' => $p->id]) . '"
                   class="btn btn-sm btn-info">Edit</a>

                <button class="btn btn-sm btn-danger deleteProduct"
                    data-id="' . $p->id . '">
                    Delete
                </button>
            ';
            })

            ->rawColumns(['images', 'status', 'action'])
            ->make(true);
    }


    public function destroy($id)
    {
        DB::beginTransaction();

        try {
            $product = Product::with([
                'variants.images',
                'variants.stocks'
            ])->findOrFail($id);

            // 1️⃣ Delete variant images (FILES + DB)
            foreach ($product->variants as $variant) {
                foreach ($variant->images as $image) {

                    // Full physical file path
                    $filePath = public_path($image->image_path);

                    if (file_exists($filePath)) {
                        unlink($filePath);
                    }

                    $image->delete();
                }

                // 2️⃣ Delete stocks
                $variant->stocks()->delete();

                // 3️⃣ Delete variant
                $variant->delete();
            }

            // 4️⃣ Detach relations
            $product->rooms()->detach();
            $product->tags()->detach();

            // 5️⃣ Delete product
            $product->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Product deleted successfully'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Delete failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    public function show($id)
    {
        $product = Product::with([
            'category',
            'subcategory',
            'variants.images',
            'variants.stocks'
        ])->findOrFail($id);

        //     $variant = $product->variants->first();
        // print_r($variant?->images);die();

        return view('admin.product.edit-product', compact('product'));
    }





public function stepOneUpdate(Request $request)
{
    // 1️⃣ Validation
    $request->validate([
        'id' => 'required|exists:products,id',
        'category_id' => 'required|exists:categories,id',
        'sub_category_id' => 'nullable|exists:sub_categories,id',
        'name' => 'required|string|max:255',
        'features' => 'nullable|string',
        'warranty_certification' => 'nullable|string',
        'maintenance' => 'nullable|string',

        // Measurement
        'length' => 'nullable|numeric',
        'width'  => 'nullable|numeric',
        'height' => 'nullable|numeric',

        'description' => 'nullable|string',
        'rooms' => 'nullable|array',
        'rooms.*' => 'exists:rooms,id',
        'tags' => 'nullable|array',
        'tags.*' => 'exists:tags,id',
        'master_catalogue_name' => 'required|string|max:255',
        'master_catalogue_code' => 'required|string|max:100',

        // ✅ Extra fields validation
        'visibility'          => 'required|in:B2C,B2B,Franchise,All',
        'gst_percent'         => 'nullable|numeric|min:0|max:100',
        'gst_inclusive'       => 'required|boolean',
        'is_featured'         => 'required|boolean',
        'related_products'    => 'nullable|array',
        'related_products.*'  => 'exists:products,id',
        'cross_sell_products' => 'nullable|array',
        'cross_sell_products.*'=> 'exists:products,id',
        'upsell_products'     => 'nullable|array',
        'upsell_products.*'   => 'exists:products,id',
        'seo_title'           => 'nullable|string|max:255',
        'seo_description'     => 'nullable|string',
        'search_keywords'     => 'nullable|string',
        'care_instructions'   => 'nullable|string',
        'fast_moving'         => 'nullable|boolean',
        'slow_moving'         => 'nullable|boolean',
    ]);

    DB::beginTransaction();

    try {
        $product = Product::findOrFail($request->id);

        // 🔹 Build measurement string
        $measurement = null;
        if ($request->length || $request->width || $request->height) {
            $measurement = trim(
                ($request->length ? $request->length . ' x ' : '') .
                ($request->width  ? $request->width  . ' x ' : '') .
                ($request->height ? $request->height : ''),
                ' x '
            );
        }

        // 2️⃣ Update product table (existing + extra fields)
        $product->update([
            'category_id' => $request->category_id,
            'sub_category_id' => $request->sub_category_id,
            'master_catalogue_name' => $request->master_catalogue_name,
            'master_catalogue_code' => $request->master_catalogue_code,
            'name' => $request->name,
            'features' => $request->features,
            'warranty_certification' => $request->warranty_certification,
            'maintenance' => $request->maintenance,
            'measurement' => $measurement,
            'description' => $request->description,

            // ✅ Extra fields
            'visibility' => $request->visibility,
            'gst_percent' => $request->gst_percent,
            'gst_inclusive' => $request->gst_inclusive,
            'is_featured' => $request->is_featured,
            'related_products' => $request->related_products,
            'cross_sell_products' => $request->cross_sell_products,
            'upsell_products' => $request->upsell_products,
            'seo_title' => $request->seo_title,
            'seo_description' => $request->seo_description,
            'search_keywords' => $request->search_keywords,
            'care_instructions' => $request->care_instructions,
            'fast_moving' => $request->fast_moving,
            'slow_moving' => $request->slow_moving,
        ]);

        // 3️⃣ Sync rooms & tags
        $product->rooms()->sync($request->rooms);
        $product->tags()->sync($request->tags ?? []);

        DB::commit();

        return response()->json([
            'success' => true,
            'message' => 'Product details updated successfully',
            'product_id' => $product->id
        ]);
    } catch (\Exception $e) {
        DB::rollBack();
        return response()->json([
            'success' => false,
            'message' => 'Update failed',
            'error' => $e->getMessage()
        ], 500);
    }
}




    public function stepTwoUpdate(Request $request)
    {
        $variantId = $request->variant_id;
        $request->validate([
            'product_id' => ['required', 'exists:products,id'],
            'sku' => [
                'required',
                'string',
                'max:255',
                Rule::unique('product_variants')
                    ->where(function ($query) use ($request) {
                        return $query->where('product_id', $request->product_id);
                    })
                    ->ignore($variantId),
            ],
            'unit' => 'required|string',
            'mrp' => 'required|numeric',
            'offer_price' => 'required|numeric',
        ]);

        DB::beginTransaction();

        try {
            $variant = ProductVariant::updateOrCreate(
                [
                    'id' => $request->variant_id,
                    'product_id' => $request->product_id,
                ],
                [
                    'sku' => $request->sku,
                    'batch_no' => $request->batch_no,
                    'unit' => $request->unit,
                    'dimensions' => $request->dimensions,
                    'material_or_fabric_composition' => $request->material_or_fabric_composition,
                    'pattern' => $request->pattern,
                    'color_name' => $request->color_name,
                    'color_code' => $request->color_code,
                    'gsm' => $request->gsm,
                    'weight' => $request->weight,
                    'rub_count' => $request->rub_count,
                    'moq' => $request->moq,
                    'mrp' => $request->mrp,
                    'offer_price' => $request->offer_price,
                ]
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Variant updated successfully',
                'product_variant_id' => $variant->id
            ]);
        } catch (\Exception $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Variant update failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    public function stepThreeUpdate(Request $request)
    {
        $request->validate([
            'product_variant_id' => 'required|exists:product_variants,id',
            'images.*' => 'image|mimes:jpg,jpeg,png,webp|max:2048',
        ]);

        $variant = ProductVariant::findOrFail($request->product_variant_id);

        if ($request->hasFile('images')) {

            foreach ($request->file('images') as $image) {

                $name = uniqid() . '.' . $image->extension();
                $image->move(public_path('uploads/products'), $name);

                ProductImages::create([
                    'product_variant_id' => $variant->id,
                    'image_path' => 'uploads/products/' . $name,
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Images updated'
        ]);
    }

    public function deleteImage(Request $request)
    {
        $image = ProductImages::findOrFail($request->image_id);

        if (file_exists(public_path($image->image_path))) {
            unlink(public_path($image->image_path));
        }

        $image->delete();

        return response()->json(['success' => true]);
    }
    // ProductController.php


public function updateStock(Request $request)
{
    $validated = $request->validate([
        'product_variant_id' => 'required|exists:product_variants,id',

        'stocks' => 'required|array|min:1',

        'stocks.*.location_id'   => 'required|exists:locations,id',

        'stocks.*.opening_qty'   => 'nullable|numeric|min:0',
        'stocks.*.received_qty'  => 'nullable|numeric|min:0',
        'stocks.*.damaged_qty'   => 'nullable|numeric|min:0',
        'stocks.*.returned_qty'  => 'nullable|numeric|min:0',

        'stocks.*.available_qty' => 'required|numeric|min:0',
        'stocks.*.reorder_level' => 'nullable|numeric|min:0',
    ]);

    $variantId = $validated['product_variant_id'];
    $stocks    = $validated['stocks'];

    // 🔹 Sort by location_id (stable updates)
    $locationIds = array_column($stocks, 'location_id');
    array_multisort($locationIds, SORT_ASC, $stocks);

    $savedIds = [];

    foreach ($stocks as $stock) {

        $model = Stock::updateOrCreate(
            [
                'product_variant_id' => $variantId,
                'location_id'        => $stock['location_id'],
            ],
            [
                'opening_qty'   => $stock['opening_qty']   ?? 0,
                'received_qty'  => $stock['received_qty']  ?? 0,
                'damaged_qty'   => $stock['damaged_qty']   ?? 0,
                'returned_qty'  => $stock['returned_qty']  ?? 0,

                'available_qty' => $stock['available_qty'],
                'reorder_level' => $stock['reorder_level'] ?? null,

                'is_available'  => $stock['available_qty'] > 0,
            ]
        );

        $savedIds[] = $model->id;
    }

    // 🔹 Remove deleted locations
    Stock::where('product_variant_id', $variantId)
        ->whereNotIn('id', $savedIds)
        ->delete();

    return response()->json([
        'success' => true,
        'message' => 'Stock updated successfully'
    ]);
}


    // StockController.php
    public function destroyStock(Stock $stock)
    {
        $stock->delete();
        return response()->json(['message' => 'Stock deleted']);
    }
    public function getProductsByCategory($category_id)
    {
        $products = Product::where('category_id', $category_id)
            ->select('id', 'name')
            ->get();

        return response()->json($products);
    }
}
